<?php

/**
 * @package     VP Neoteric
 *
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2023-2025 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Uri\Uri;
use Virtueplanet\Plugin\System\Prime\Helper\GlobalHelper;
use Virtueplanet\Plugin\System\Prime\Helper\GoogleFontHelper;
use Virtueplanet\Plugin\System\Prime\Helper\TemplateHelper;

/** @var Joomla\CMS\Document\HtmlDocument $this */

/** @var \Joomla\CMS\Application\SiteApplication $app */
$app  = Factory::getApplication();
$user = $app->getIdentity();
$wa   = $this->getWebAssetManager();

/** @var Joomla\Registry\Registry $params */
$params = $this->params;

// Browsers support SVG favicons
$faviconSvg = $params->get('favicon_svg', '');

if (!empty($faviconSvg)) {
    $this->addHeadLink(Uri::root(true) . '/images/' . $faviconSvg, 'icon', 'rel', ['type' => 'image/svg+xml']);
} else {
    $this->addHeadLink(HTMLHelper::_('image', 'joomla-favicon.svg', '', [], true, 1), 'icon', 'rel', ['type' => 'image/svg+xml']);
}

// Add ICO favicon.
$favicon = $params->get('favicon', '');

if (!empty($favicon)) {
    $this->addHeadLink(Uri::root(true) . '/images/' . $favicon, 'alternate icon', 'rel', ['type' => 'image/vnd.microsoft.icon']);
} else {
    $this->addHeadLink(HTMLHelper::_('image', 'favicon.ico', '', [], true, 1), 'alternate icon', 'rel', ['type' => 'image/vnd.microsoft.icon']);
}

// Add Pinned Tab Icon
$pinnedTabIcon      = $params->get('pinned_tab_icon', '');
$pinnedTabIconColor = $params->get('pinned_tab_icon_color', '#000000');

if (!empty($pinnedTabIcon)) {
    $this->addHeadLink(Uri::root(true) . '/images/' . $pinnedTabIcon, 'mask-icon', 'rel', ['color' => $pinnedTabIconColor]);
} else {
    $this->addHeadLink(HTMLHelper::_('image', 'joomla-favicon-pinned.svg', '', [], true, 1), 'mask-icon', 'rel', ['color' => $pinnedTabIconColor]);
}

// Detecting Active Variables
$option    = strtolower($app->input->getCmd('option', ''));
$view      = strtolower($app->input->getCmd('view', ''));
$layout    = strtolower($app->input->getCmd('layout', ''));
$task      = strtolower($app->input->getCmd('task', ''));
$itemid    = $app->input->getInt('Itemid', 0);
$tmpl      = strtolower($app->input->getCmd('tmpl', ''));
$manage    = $app->input->getInt('manage', 0);
$sitename  = htmlspecialchars($app->get('sitename'), ENT_QUOTES, 'UTF-8');
$menu      = $app->getMenu()->getActive();
$pageclass = $menu !== null ? $menu->getParams()->get('pageclass_sfx', '') : '';
$version   = TemplateHelper::getVersionFromManifest(__DIR__ . '/templateDetails.xml');
$template  = $app->getTemplate(false);
$bodyClass = ($option ? ' ' . str_replace('_', '-', $option) : '') .
    ($view ? ' view-' . $view : ' no-view') .
    ($layout ? ' layout-' . $layout : ' no-layout') .
    ($task ? ' task-' . $task : ' no-task') .
    ($itemid ? ' itemid-' . $itemid : '') .
    ($pageclass ? ' ' . $pageclass : '') .
    ($this->direction === 'rtl' ? ' rtl' : ' ltr') .
    (' tpl-' . str_replace('_', '-', $template)) .
    (' v-' . $version) .
    (' j-' . substr(JVERSION, 0, 1));

// Container class.
$containerClass = $params->get('fluid_container', 0) ? 'container-fluid' : 'container-xxl';

// Additional header class.
$stickyHeader = $params->get('sticky_header', 0) ? ' sticky-top' : '';

// Check for full width views.
$isFullWidth = TemplateHelper::isFullWidthView($params->get('fullwidth_views', []));

// Check for full width menu items.
$fullWidthMenuItems = $params->get('fullwidth_itemids', []);

if (!$isFullWidth && !empty($fullWidthMenuItems) && $itemid > 0 && in_array($itemid, $fullWidthMenuItems)) {
    $isFullWidth = true;
}

// Build left, right and main grid class.
$leftGridSize  = (int) $params->get('left_grid', 3);
$rightGridSize = (int) $params->get('right_grid', 3);
$mainGridSize  = (int) $params->get('main_grid', 6);

// Adjusting content width
if ($isFullWidth) {
    $mainClass  = 'col-12';
    $leftClass  = '';
    $rightClass = '';
} elseif ($this->countModules('left', true) && $this->countModules('right', true)) {
    $validLeftSize  = TemplateHelper::getValidGridSize($leftGridSize, 'lg');
    $validRightSize = TemplateHelper::getValidGridSize($rightGridSize, 'lg');
    $validMainSize  = TemplateHelper::getValidGridSize($mainGridSize, 'lg');

    $mainClass  = 'col-12 ' . $validMainSize;
    $leftClass  = 'col-12 ' . $validLeftSize;
    $rightClass = 'col-12 ' . $validRightSize;

    if (($params->get('left_module_position', 'below') == 'below')) {
        $mainClass  .= ' order-1 order-lg-2';
        $leftClass  .= ' order-2 order-lg-1';
        $rightClass .= ' order-3 order-lg-3';
    }
} elseif ($this->countModules('left', true) && !$this->countModules('right', true)) {
    $validLeftSize = TemplateHelper::getValidGridSize($leftGridSize, 'lg');
    $validMainSize = TemplateHelper::getValidGridSize(($mainGridSize + $rightGridSize), 'lg');

    $mainClass  = 'col-12 ' . $validMainSize;
    $leftClass  = 'col-12 ' . $validLeftSize;
    $rightClass = '';

    if ($params->get('left_module_position', 'below') == 'below') {
        $mainClass .= ' order-1 order-lg-2';
        $leftClass .= ' order-2 order-lg-1';
    }
} elseif (!$this->countModules('left', true) && $this->countModules('right', true)) {
    $validRightSize = TemplateHelper::getValidGridSize($rightGridSize, 'lg');
    $validMainSize  = TemplateHelper::getValidGridSize(($mainGridSize + $leftGridSize), 'lg');

    $mainClass  = 'col-12 ' . $validMainSize;
    $leftClass  = '';
    $rightClass = 'col-12 ' . $validRightSize;
} else {
    $mainClass  = 'col-12';
    $leftClass  = '';
    $rightClass = '';
}

// Use a font scheme if set in the template style options
$paramsFontScheme = $params->get('use_font_scheme', 'media/templates/site/vp_neoteric/css/global/fonts-local_roboto.css');

if ($paramsFontScheme) {
    if ($paramsFontScheme === 'google') {
        $bodyFontName       = $params->get('google_font_name', '');
        $bodyFontVariant    = $params->get('google_font_variant', '');
        $addFontVariants    = $params->get('google_font_additional_variants', []);
        $headingFontName    = $params->get('heading_google_font_name', '');
        $headingFontVariant = $params->get('heading_google_font_variant', '');
        $gfontsJsonFile     = 'media/templates/site/vp_neoteric/webfonts/gfonts.json';
        $fontStyles         = '';

        if ($bodyFontName || $headingFontName) {
            $this->getPreloadManager()->preconnect('https://fonts.googleapis.com/', ['crossorigin' => 'anonymous']);
            $this->getPreloadManager()->preconnect('https://fonts.gstatic.com/', ['crossorigin' => 'anonymous']);

            if ($bodyFontName) {
                GoogleFontHelper::addFontFamily($bodyFontName, $bodyFontVariant);

                if (!empty($addFontVariants)) {
                    GoogleFontHelper::addFontFamily($bodyFontName, $addFontVariants);
                }

                $fontInfo = GoogleFontHelper::getFontInfo($bodyFontName, $bodyFontVariant, $gfontsJsonFile);

                if ($fontInfo) {
                    $fontStyles .= '--neoteric-font-family-body: ' . $fontInfo->family . ";\n";
                    $fontStyles .= '--neoteric-font-weight-normal: ' . $fontInfo->weight . ";\n";
                    $fontStyles .= '--neoteric-font-style-normal: ' . $fontInfo->style . ";\n";
                }
            }

            if ($headingFontName) {
                GoogleFontHelper::addFontFamily($headingFontName, $headingFontVariant);

                $fontInfo = GoogleFontHelper::getFontInfo($headingFontName, $headingFontVariant, $gfontsJsonFile);

                if ($fontInfo) {
                    $fontStyles .= '--neoteric-font-family-headings: ' . $fontInfo->family . ";\n";
                    $fontStyles .= '--neoteric-font-weight-headings: ' . $fontInfo->weight . ";\n";
                    $fontStyles .= '--neoteric-font-style-headings: ' . $fontInfo->style . ";\n";
                }
            }

            $fontStyleUrl = GoogleFontHelper::getAddedFontsUrl();

            if ($fontStyleUrl) {
                $this->getPreloadManager()->preload($fontStyleUrl, ['as' => 'style', 'crossorigin' => 'anonymous']);

                if ($params->get('lazyload_google_fonts', 1)) {
                    $wa->registerAndUseStyle('fontscheme.current', $fontStyleUrl, [], ['media' => 'print', 'rel' => 'lazy-stylesheet', 'onload' => 'this.media=\'all\'', 'crossorigin' => 'anonymous']);
                } else {
                    $wa->registerAndUseStyle('fontscheme.current', $fontStyleUrl, [], ['media' => 'all', 'crossorigin' => 'anonymous']);
                }
            }
        }

        if (!empty($fontStyles)) {
            $wa->addInlineStyle(":root {\n" . $fontStyles . "}");
        }
    } else {
        $wa->registerAndUseStyle('fontscheme.current', $paramsFontScheme, ['version' => 'auto'], ['media' => 'all']);
        $this->getPreloadManager()->preload($wa->getAsset('style', 'fontscheme.current')->getUri() . '?' . $this->getMediaVersion(), ['as' => 'style']);
    }
}

// Enable assets
$wa->usePreset('template.vp_neoteric.' . ($this->direction === 'rtl' ? 'rtl' : 'ltr'))
    ->useStyle('template.active.language')
    ->useStyle('template.user')
    ->useScript('template.user');

$cssVariables    = TemplateHelper::getCSSVariables($params, __DIR__, 'style');
$cssVariablesStr = '';

foreach ($cssVariables as $name => $value) {
    $cssVariablesStr .= $name . ': ' . $value . ';' . "\n";
}

$cssVariablesStr = ':root {' . "\n" . $cssVariablesStr . '}';

$wa->addInlineStyle($cssVariablesStr);

// Override 'template.active' asset to set correct ltr/rtl dependency
$wa->registerStyle('template.active', '', [], [], ['template.vp_neoteric.' . ($this->direction === 'rtl' ? 'rtl' : 'ltr')]);

// Load language strings for scripts
Text::script('WARNING');
Text::script('COM_VIRTUEMART_PRODUCT_ADDED_SUCCESSFULLY');
Text::script('COM_VIRTUEMART_MINICART_ERROR_JS');

// Logo file or site title param
if ($params->get('logo_file')) {
    $logo = HTMLHelper::_('image', Uri::root(false) . htmlspecialchars($this->params->get('logo_file'), ENT_QUOTES), $sitename, ['class' => 'logo d-inline-block', 'loading' => 'eager', 'decoding' => 'async'], false, 0);
} elseif ($params->get('site_title')) {
    $logo = '<span title="' . $sitename . '">' . htmlspecialchars($params->get('site_title'), ENT_COMPAT, 'UTF-8') . '</span>';
} else {
    $logo = HTMLHelper::_('image', 'logo.svg', $sitename, ['class' => 'logo d-inline-block', 'loading' => 'eager', 'decoding' => 'async'], true, 0);
}

// Small logo file or site title param
if ($params->get('logo_file_small')) {
    $smallLogo = HTMLHelper::_('image', Uri::root(false) . htmlspecialchars($this->params->get('logo_file_small'), ENT_QUOTES), $sitename, ['class' => 'logo d-inline-block', 'loading' => 'eager', 'decoding' => 'async'], false, 0);
} elseif ($params->get('logo_file')) {
    $smallLogo = '<img src="' . Uri::root(true) . '/' . htmlspecialchars($params->get('logo_file'), ENT_QUOTES) . '" loading="eager" alt="' . $sitename . '">';
} elseif ($params->get('site_title')) {
    $smallLogo = '<span title="' . $sitename . '">' . htmlspecialchars($params->get('site_title'), ENT_COMPAT, 'UTF-8') . '</span>';
} else {
    $smallLogo = HTMLHelper::_('image', 'logo-small.svg', $sitename, ['class' => 'logo d-inline-block', 'loading' => 'eager', 'decoding' => 'async'], true, 0);
}

if ($params->get('show_login_register', 1)) {
    GlobalHelper::getInstance()->setData('showOffcanvasLogin', true);
}

// Add viewport
$this->setMetaData('viewport', 'width=device-width, initial-scale=1');

// Theme Color
$this->setMetaData('theme-color', $params->get('meta_theme_color', '#0088cc'));

// Alter viewport for iPhone
$wa->addInlineScript("
    document.addEventListener('DOMContentLoaded', function() {
        var userAgent = navigator.userAgent;
        if((userAgent.indexOf('iPhone') > -1 || userAgent.indexOf('iPad') > -1 || userAgent.indexOf('iPod') > -1) && document.querySelector('[name=viewport]')) {
            document
              .querySelector('[name=viewport]')
              .setAttribute('content', 'width=device-width, initial-scale=1, maximum-scale=1');

            document.documentElement.classList.add('is-ios');
        }
    });
");

// Fix iOS 26 layout issue until Apple fixes it in iOS (Beta Feature)
if ($params->get('fix_ios_26_layout_issue', 1)) {
    $wa->addInlineStyle("
        html.is-ios {
            height: 100%;
            overflow: hidden;
        }
        html.is-ios body {
            height: 100%;
            overflow: auto;
            -webkit-overflow-scrolling: touch;
        }
    ");
}

// Defer fontawesome for increased performance. Once the page is loaded javascript changes it to a stylesheet.
if ($params->get('lazyload_fontawesome', 1)) {
    $wa->getAsset('style', 'fontawesome')->setAttribute('rel', 'lazy-stylesheet');
    $this->getPreloadManager()->preload($wa->getAsset('style', 'fontawesome')->getUri() . '?' . $this->getMediaVersion(), ['as' => 'style']);
}

if ($params->get('lazyload_icon_fonts', 0)) {
    $wa->getAsset('style', 'template.vp_neoteric.icons')->setAttribute('rel', 'lazy-stylesheet');
    $this->getPreloadManager()->preload($wa->getAsset('style', 'template.vp_neoteric.icons')->getUri() . '?' . $this->getMediaVersion(), ['as' => 'style']);
}

// Unload unnecessary assets except when VirtueMart's front end editing mode.
if (($option === 'com_virtuemart' && $manage) === false) {
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vm-ltr-common.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vm-ltr-site.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vm-ltr-reviews.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vmpanels.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vmsite-ltr.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vmsite-rtl.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vm-bs3-common.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vm-bs4-common.css');
    TemplateHelper::enqueueToUnloadAssets('styleSheets', 'components/com_virtuemart/assets/css/vm-bs5-common.css');
    TemplateHelper::enqueueToUnloadAssets('scripts', 'components/com_virtuemart/assets/js/cvfind.js');
}

// Unload unnecessary assets loaded by VirtueMart Essentials
TemplateHelper::enqueueToUnloadAssets('styleSheets', 'media/com_vmessentials/css/font-awesome.css');
TemplateHelper::enqueueToUnloadAssets('styleSheets', 'media/com_vmessentials/css/font-awesome.min.css');
TemplateHelper::enqueueToUnloadAssets('styleSheets', 'media/com_vmessentials/css/iframe-modal.css');
TemplateHelper::enqueueToUnloadAssets('styleSheets', 'media/com_vmessentials/css/iframe-modal.min.css');
TemplateHelper::enqueueToUnloadAssets('scripts', 'media/com_vmessentials/js/iframe-modal.js');
TemplateHelper::enqueueToUnloadAssets('scripts', 'media/com_vmessentials/js/iframe-modal.min.js');

// Change VirtueMart Essentials Configuration.
$comVMEssentialsParams = ComponentHelper::getParams('com_vmessentials');
$comVMEssentialsParams->set('auto_buttons', 0);

// Prepare custom codes.
$customCodeBeforeHead = $params->get('custom_codes_before_head', '');
$customCodeAfterBody  = $params->get('custom_codes_after_body', '');
$customCodeBeforeBody = $params->get('custom_codes_before_body', '');
$customCSS            = $params->get('custom_css', '');
$customCSS            = trim($customCSS);
$customCSS            = !empty($customCSS) ? '<style>' . "\n" . $customCSS . "\n" . '</style>' : '';
$customJavascript     = $params->get('custom_js', '');
$customJavascript     = trim($customJavascript);
$customJavascript     = !empty($customJavascript) ? '<script>' . "\n" . $customJavascript . "\n" . '</script>' : '';

// Get product navigation.
$productNavigation = '';

if ($option === 'com_virtuemart' && $view === 'productdetails') {
    $productNavigation = GlobalHelper::getInstance()->getData('product-navigation', $productNavigation);
}

$showLogin = GlobalHelper::getInstance()->getData('showOffcanvasLogin', false);
?>
<!DOCTYPE html>
<html lang="<?php echo $this->language; ?>" dir="<?php echo $this->direction; ?>">

<head>
    <jdoc:include type="metas" />
    <jdoc:include type="styles" />
    <jdoc:include type="scripts" />
    <?php echo $customCodeBeforeHead; ?>
    <?php echo $customCSS; ?>
</head>

<body class="site<?php echo $bodyClass; ?>">
    <?php echo $customCodeAfterBody; ?>
    <?php if ($this->countModules('top-left', true) || $this->countModules('top-right', true)) : ?>
        <div class="top-bar-content">
            <div class="<?php echo $containerClass; ?>">
                <div class="container-inner">
                    <div class="d-flex flex-column flex-md-row justify-content-md-between align-items-center gap-md-4">
                        <?php if ($this->countModules('top-left', true)) : ?>
                            <div class="top-left d-flex flex-row align-items-stretch justify-content-center justify-content-md-start gap-3">
                                <jdoc:include type="modules" name="top-left" style="inline" />
                            </div>
                        <?php endif; ?>
                        <?php if ($this->countModules('top-right', true)) : ?>
                            <div class="top-right d-flex flex-row align-items-stretch justify-content-center justify-content-md-end gap-3 flex-md-grow-1">
                                <jdoc:include type="modules" name="top-right" style="inline" />
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>
    <header class="header-content<?php echo $stickyHeader; ?>">
        <div class="header-main">
            <div class="<?php echo $containerClass; ?>">
                <div class="container-inner">
                    <div class="d-flex d-lg-none flex-row flex-nowrap justify-content-between align-items-center gap-4">
                        <div class="mobile-header-left d-flex flex-row justify-content-start align-items-center">
                            <button type="button" class="offcanvas-toggler btn-icon" data-bs-toggle="offcanvas" data-bs-target="#offcanvas" aria-controls="offcanvas" aria-label="<?php echo htmlspecialchars(Text::_('TPL_VP_NEOTERIC_TOGGLE_NAVIGATION'), ENT_QUOTES, 'UTF-8'); ?>">
                                <i class="vpi vpi-bars vpi-2x" aria-hidden="true"></i>
                            </button>
                        </div>
                        <div class="mobile-header-center d-flex flex-row justify-content-center align-items-center">
                            <a class="brand-logo" href="<?php echo $this->baseurl; ?>/">
                                <?php echo $smallLogo; ?>
                            </a>
                        </div>
                        <div class="mobile-header-right d-flex flex-row justify-content-end align-items-center">
                            <?php if ($this->countModules('search', true)) : ?>
                                <a href="#headerSearch" class="d-inline-block link-inherit-color btn-icon" role="button" aria-controls="headerSearch" aria-label="<?php echo htmlspecialchars(Text::_('JSEARCH_FILTER'), ENT_QUOTES, 'UTF-8'); ?>" data-tmpl-toggle="search">
                                    <i class="vpi vpi-search vpi-lg" aria-hidden="true"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="d-block d-lg-flex flex-row flex-nowrap justify-content-between align-items-center gap-5">
                        <div class="header-left d-none d-lg-flex align-items-center justify-content-start">
                            <div class="site-logo">
                                <a class="brand-logo" href="<?php echo $this->baseurl; ?>/">
                                    <?php echo $logo; ?>
                                </a>
                            </div>
                        </div>
                        <?php if ($this->countModules('search', true)) : ?>
                            <div id="headerSearch" class="header-search-container flex-grow-1">
                                <div class="header-search-inner d-flex justify-content-start align-items-center gap-2">
                                    <a href="#headerSearch" class="d-inline-block d-lg-none link-inherit-color py-1 pe-2" role="button" aria-controls="headerSearch" aria-label="<?php echo htmlspecialchars(Text::_('JCLOSE'), ENT_QUOTES, 'UTF-8'); ?>" data-tmpl-toggle="search">
                                        <i class="vpi vpi-arrow-left vpi-2x" aria-hidden="true"></i>
                                    </a>
                                    <div class="flex-grow-1">
                                        <jdoc:include type="modules" name="search" style="none" />
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                        <div class="header-right-lg footer-toolbar fixed-bottom d-flex flex-row justify-content-around justify-content-lg-end align-items-center">
                            <div class="home-page-link-wrapper d-lg-none">
                                <a href="<?php echo $this->baseurl; ?>/" class="d-inline-block link-inherit-color btn-icon" aria-label="<?php echo htmlspecialchars(Text::_('TPL_VP_NEOTERIC_HOME_PAGE'), ENT_QUOTES, 'UTF-8'); ?>">
                                    <i class="vpi vpi-home vpi-lg" aria-hidden="true"></i>
                                </a>
                            </div>
                            <?php if ($params->get('show_login_register', 1)) : ?>
                                <?php if ($user->guest) : ?>
                                    <a class="d-inline-block link-inherit-color btn-icon text-uppercase" href="#" data-bs-toggle="offcanvas" data-bs-target="#offcanvasLogin" aria-controls="offcanvasLogin" aria-label="<?php echo Text::_('JLOGIN'); ?> / <?php echo Text::_('JREGISTER'); ?>">
                                        <i class="vpi vpi-user vpi-lg d-lg-none" aria-hidden="true"></i>
                                        <span class="d-none d-lg-inline"><?php echo Text::_('JLOGIN'); ?>&nbsp;/&nbsp;<?php echo Text::_('JREGISTER'); ?></span>
                                    </a>
                                <?php else : ?>
                                    <a class="d-inline-block link-inherit-color btn-icon text-uppercase" href="#" data-bs-toggle="offcanvas" data-bs-target="#offcanvasLogin" aria-controls="offcanvasLogin" aria-label="<?php echo Text::_('TPL_VP_NEOTERIC_MY_ACCOUNT'); ?>">
                                        <i class="vpi vpi-user vpi-lg d-lg-none" aria-hidden="true"></i>
                                        <span class="d-none d-lg-inline"><?php echo Text::_('TPL_VP_NEOTERIC_MY_ACCOUNT'); ?></span>
                                    </a>
                                <?php endif; ?>
                            <?php endif; ?>
                            <jdoc:include type="modules" name="wishlist" style="none" />
                            <jdoc:include type="modules" name="compare" style="none" />
                            <jdoc:include type="modules" name="mini-cart" style="none" />
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($this->countModules('category-dropdown', true) || $this->countModules('mega-menu', true)) : ?>
            <div id="mega-menu" class="mega-menu-container d-none d-lg-block">
                <div class="<?php echo $containerClass; ?>">
                    <div class="container-inner">
                        <div class="d-flex flex-row justify-content-start align-items-center">
                            <jdoc:include type="modules" name="category-dropdown" style="none" />
                            <jdoc:include type="modules" name="mega-menu" style="none" />
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </header>

    <div class="site-content">
        <?php if ($this->countModules('slider', true)) : ?>
            <div class="slider-section">
                <jdoc:include type="modules" name="slider" style="basic" />
            </div>
        <?php endif; ?>

        <?php if ($this->countModules('breadcrumbs', true) || !empty($productNavigation)) : ?>
            <div class="breadcrumbs-section">
                <div class="<?php echo $containerClass; ?>">
                    <div class="container-inner">
                        <div class="breadcrumbs-inner d-flex gap-2 align-items-center">
                            <jdoc:include type="modules" name="breadcrumbs" style="basic" />
                            <?php echo $productNavigation; ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <div class="main-content-area">
            <div class="<?php echo $containerClass; ?>">
                <div class="container-inner">
                    <?php if ($this->countModules('main-top-vertical', true)) : ?>
                        <div class="main-top-vertical-section vertical-module-section">
                            <jdoc:include type="modules" name="main-top-vertical" style="block" />
                        </div>
                    <?php endif; ?>

                    <?php if ($this->countModules('main-top-horizontal', true)) : ?>
                        <div class="main-top-horizontal-section">
                            <div class="row">
                                <jdoc:include type="modules" name="main-top-horizontal" style="row" />
                            </div>
                        </div>
                    <?php endif; ?>

                    <div class="main-section">
                        <div class="row">
                            <?php if (!empty($leftClass)) : ?>
                                <aside id="left-aside" class="left-aside <?php echo $leftClass; ?>">
                                    <div class="aside-inner">
                                        <jdoc:include type="modules" name="left" style="block" />
                                    </div>
                                </aside>
                            <?php endif; ?>

                            <div id="content" class="<?php echo $mainClass; ?>">
                                <div class="content-inner">
                                    <?php if ($this->countModules('content-top-vertical', true)) : ?>
                                        <div class="content-top-vertical-section vertical-module-section">
                                            <jdoc:include type="modules" name="content-top-vertical" style="block" />
                                        </div>
                                    <?php endif; ?>
                                    <?php if ($this->countModules('content-top-horizontal', true)) : ?>
                                        <div class="content-top-horizontal-section">
                                            <div class="row">
                                                <jdoc:include type="modules" name="content-top-horizontal" style="row" />
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                    <jdoc:include type="message" />
                                    <main>
                                        <jdoc:include type="component" />
                                    </main>
                                    <?php if ($this->countModules('content-bottom-vertical', true)) : ?>
                                        <div class="content-bottom-vertical-section vertical-module-section">
                                            <jdoc:include type="modules" name="content-bottom-vertical" style="block" />
                                        </div>
                                    <?php endif; ?>
                                    <?php if ($this->countModules('content-bottom-horizontal', true)) : ?>
                                        <div class="content-bottom-horizontal-section">
                                            <div class="row">
                                                <jdoc:include type="modules" name="content-bottom-horizontal" style="row" />
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>

                            <?php if (!empty($rightClass)) : ?>
                                <aside id="right-aside" class="right-aside <?php echo $rightClass; ?>">
                                    <div class="aside-inner">
                                        <jdoc:include type="modules" name="right" style="block" />
                                    </div>
                                </aside>
                            <?php endif; ?>
                        </div>
                    </div>

                    <?php if ($this->countModules('main-bottom-vertical', true)) : ?>
                        <div class="main-bottom-vertical-section vertical-module-section">
                            <jdoc:include type="modules" name="main-bottom-vertical" style="block" />
                        </div>
                    <?php endif; ?>

                    <?php if ($this->countModules('main-bottom-horizontal', true)) : ?>
                        <div class="main-bottom-horizontal-section">
                            <div class="row">
                                <jdoc:include type="modules" name="main-bottom-horizontal" style="row" />
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <?php if ($this->countModules('utility-wide', true)) : ?>
            <div class="utility-wide-section">
                <jdoc:include type="modules" name="utility-wide" style="basic" />
            </div>
        <?php endif; ?>

        <?php if ($this->countModules('utility-vertical', true)) : ?>
            <div class="utility-vertical-section vertical-module-section">
                <div class="<?php echo $containerClass; ?>">
                    <div class="container-inner">
                        <jdoc:include type="modules" name="utility-vertical" style="block" />
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($this->countModules('utility-horizontal', true)) : ?>
            <div class="utility-horizontal-section">
                <div class="<?php echo $containerClass; ?>">
                    <div class="container-inner">
                        <div class="row">
                            <jdoc:include type="modules" name="utility-horizontal" style="row" />
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($this->countModules('bottom-slider', true)) : ?>
            <div class="bottom-slider-section">
                <div class="<?php echo $containerClass; ?>">
                    <div class="container-inner">
                        <jdoc:include type="modules" name="bottom-slider" style="basic" />
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($this->countModules('bottom-vertical', true) || $this->countModules('bottom-horizontal', true)) : ?>
            <div class="bottom-section">
                <div class="<?php echo $containerClass; ?>">
                    <div class="container-inner">
                        <?php if ($this->countModules('bottom-vertical', true)) : ?>
                            <div class="bottom-vertical-section vertical-module-section">
                                <jdoc:include type="modules" name="bottom-vertical" style="block" />
                            </div>
                        <?php endif; ?>
                        <?php if ($this->countModules('bottom-horizontal', true)) : ?>
                            <div class="bottom-horizontal-section">
                                <div class="row">
                                    <jdoc:include type="modules" name="bottom-horizontal" style="row" />
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <?php if ($this->countModules('footer-left', true) || $this->countModules('footer-right', true)) : ?>
        <footer class="footer-area">
            <div class="<?php echo $containerClass; ?>">
                <div class="container-inner">
                    <div class="row">
                        <?php if ($this->countModules('footer-left', true)) : ?>
                            <div class="<?php echo $this->countModules('footer-right', true) ? 'col-lg-6 mb-4 mb-lg-0' : 'col-lg-12'; ?> footer-left-section d-flex justify-content-center justify-content-lg-start text-center text-lg-start">
                                <jdoc:include type="modules" name="footer-left" style="inline" />
                            </div>
                        <?php endif; ?>
                        <?php if ($this->countModules('footer-right', true)) : ?>
                            <div class="<?php echo $this->countModules('footer-left', true) ? 'col-lg-6' : 'col-lg-12'; ?> footer-right-section d-flex justify-content-center justify-content-lg-end text-center text-lg-end">
                                <jdoc:include type="modules" name="footer-right" style="inline" />
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </footer>
    <?php endif; ?>

    <?php if ($params->get('back_to_top')) : ?>
        <a href="#top" id="back-top" class="back-to-top-link" aria-label="<?php echo Text::_('TPL_VP_NEOTERIC_BACKTOTOP'); ?>">
            <span class="icon-arrow-up icon-fw" aria-hidden="true"></span>
        </a>
    <?php endif; ?>

    <?php echo LayoutHelper::render('blocks.offcanvas', ['document' => $this, 'params' => $params, 'elementId' => 'offcanvas']); ?>

    <?php if ($showLogin) : ?>
        <?php echo LayoutHelper::render('blocks.login', ['document' => $this, 'params' => $params, 'elementId' => 'offcanvasLogin']); ?>
    <?php endif; ?>

    <?php if ($params->get('show_cookie_consent', 1)) : ?>
        <?php echo LayoutHelper::render('blocks.cookie_consent', ['position' => $params->get('cookie_consent_position'), 'policyArticleId' => (int) $params->get('cookie_policy_article_id'), 'tenure' => (int) $params->get('cookie_consent_tenure', 30)]); ?>
    <?php endif; ?>

    <jdoc:include type="modules" name="debug" style="none" />

    <?php echo $customCodeBeforeBody; ?>
    <?php echo $customJavascript; ?>
</body>

</html>
