<?php

/**
 * @package     VP Neoteric
 *
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2023-2025 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * This is a file to add template specific chrome to pagination rendering.
 *
 * pagination_list_footer
 *     Input variable $list is an array with offsets:
 *         $list[limit]        : int
 *         $list[limitstart]   : int
 *         $list[total]        : int
 *         $list[limitfield]   : string
 *         $list[pagescounter] : string
 *         $list[pageslinks]   : string
 *
 * pagination_list_render
 *     Input variable $list is an array with offsets:
 *         $list[all]
 *             [data]      : string
 *             [active]    : boolean
 *         $list[start]
 *             [data]      : string
 *             [active]    : boolean
 *         $list[previous]
 *             [data]      : string
 *             [active]    : boolean
 *         $list[next]
 *             [data]      : string
 *             [active]    : boolean
 *         $list[end]
 *             [data]      : string
 *             [active]    : boolean
 *         $list[pages]
 *             [{PAGE}][data]      : string
 *             [{PAGE}][active]    : boolean
 *
 * pagination_item_active
 *     Input variable $item is an object with fields:
 *         $item->base    : integer
 *         $item->link    : string
 *         $item->text    : string
 *
 * pagination_item_inactive
 *     Input variable $item is an object with fields:
 *         $item->base    : integer
 *         $item->link    : string
 *         $item->text    : string
 *
 * This gives template designers ultimate control over how pagination is rendered.
 *
 * NOTE: If you override pagination_item_active OR pagination_item_inactive you MUST override them both
 */

/**
 * Renders the pagination footer
 *
 * @param   array  $list  Array containing pagination footer
 *
 * @return  string  HTML markup for the full pagination footer
 *
 * @since   3.0
 */
function pagination_list_footer($list)
{
    $html = "<div class=\"pagination-container\">\n";
    $html .= $list['pageslinks'];
    $html .= "\n<input type=\"hidden\" name=\"" . $list['prefix'] . "limitstart\" value=\"" . $list['limitstart'] . "\" />";
    $html .= "\n</div>";

    return $html;
}

/**
 * Renders the pagination list
 *
 * @param   array  $list  Array containing pagination information
 *
 * @return  string  HTML markup for the full pagination object
 *
 * @since   3.0
 */
function pagination_list_render($list)
{
    // Calculate to display range of pages
    $currentPage = 1;
    $range = 1;
    $step = 5;
    foreach ($list['pages'] as $k => $page) {
        if (!$page['active']) {
            $currentPage = $k;
        }
    }
    if ($currentPage >= $step) {
        if ($currentPage % $step == 0) {
            $range = ceil($currentPage / $step) + 1;
        } else {
            $range = ceil($currentPage / $step);
        }
    }

    $html = '<ul class="pagination m-0">';
    $html .= $list['start']['data'];
    $html .= $list['previous']['data'];

    foreach ($list['pages'] as $k => $page) {
        if (in_array($k, range($range * $step - ($step + 1), $range * $step))) {
            if (($k % $step == 0 || $k == $range * $step - ($step + 1)) && $k != $currentPage && $k != $range * $step - $step) {
                $page['data'] = preg_replace('#(<a.*?>).*?(</a>)#', '$1...$2', $page['data']);
            }
        }

        $html .= $page['data'];
    }

    $html .= $list['next']['data'];
    $html .= $list['end']['data'];

    $html .= '</ul>';
    return $html;
}

/**
 * Renders an active item in the pagination block
 *
 * @param   JPaginationObject  $item  The current pagination object
 *
 * @return  string  HTML markup for active item
 *
 * @since   3.0
 */
function pagination_item_active(&$item)
{
    $app     = Factory::getApplication();
    $display = $item->text;
    $class   = '';

    switch ((string) $item->text) {
        // Check for "Start" item
        case Text::_('JLIB_HTML_START'):
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-double-right' : 'icon-angle-double-left';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

            // Check for "Prev" item
        case $item->text === Text::_('JPREV'):
            $item->text = Text::_('JPREVIOUS');
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-right' : 'icon-angle-left';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

            // Check for "Next" item
        case Text::_('JNEXT'):
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-left' : 'icon-angle-right';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

            // Check for "End" item
        case Text::_('JLIB_HTML_END'):
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-double-left' : 'icon-angle-double-right';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

        default:
            $icon = null;
            $aria = Text::sprintf('JLIB_HTML_GOTO_PAGE', strtolower($item->text));
            break;
    }

    if ($icon !== null) {
        $display = '<span class="' . $icon . '" aria-hidden="true"></span>';
    }

    return '<li class="page-item' . $class . '"><a title="' . $item->text . '" href="' . $item->link . '" class="page-link" aria-label="' . $aria . '">' . $display . '</a></li>';
}

/**
 * Renders an inactive item in the pagination block
 *
 * @param   JPaginationObject  $item  The current pagination object
 *
 * @return  string  HTML markup for inactive item
 *
 * @since   3.0
 */
function pagination_item_inactive(&$item)
{
    $app     = Factory::getApplication();
    $display = $item->text;
    $input   = $app->input;
    $isPage  = false;

    switch ((string) $item->text) {
        // Check for "Start" item
        case Text::_('JLIB_HTML_START'):
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-double-right' : 'icon-angle-double-left';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

            // Check for "Prev" item
        case $item->text === Text::_('JPREV'):
            $item->text = Text::_('JPREVIOUS');
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-right' : 'icon-angle-left';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

            // Check for "Next" item
        case Text::_('JNEXT'):
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-left' : 'icon-angle-right';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

            // Check for "End" item
        case Text::_('JLIB_HTML_END'):
            $icon = $app->getLanguage()->isRtl() ? 'icon-angle-double-left' : 'icon-angle-double-right';
            $aria = Text::sprintf('JLIB_HTML_GOTO_POSITION', strtolower($item->text));
            break;

        default:
            $icon   = null;
            $aria   = Text::sprintf('JLIB_HTML_GOTO_PAGE', strtolower($item->text));
            $isPage = true;
            break;
    }

    if ($icon !== null) {
        $display = '<span class="' . $icon . '" aria-hidden="true"></span>';
    }

    if (((property_exists($item, 'active') && $item->active)) || ($app->isClient('site') && $input->getCmd('option') == 'com_virtuemart' && $isPage)) {
        $aria = Text::sprintf('JLIB_HTML_PAGE_CURRENT', strtolower($item->text));

        return '<li class="page-item active"><a title="' . $item->text . '" href="#" class="page-link" aria-label="' . $aria . '" aria-current="true">' . $display . '</a></li>';
    } else {
        return '<li class="page-item disabled"><span class="page-link" aria-label="' . $aria . '">' . $display . '</span></li>';
    }
}
