<?php

/**
 * @package     VP Prime Framework
 *
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2012-2025 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Virtueplanet\Plugin\System\Prime\Service\HTML;

use Joomla\CMS\Factory;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\Utilities\ArrayHelper;
use Virtueplanet\Plugin\System\Prime\Helper\VirtuemartHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * HTML class for VirtueMart.
 */
abstract class Virtuemart
{
    protected static $categoryOptions = [];

    public static function categoryOptions($parentId = 0, $level = 0, $vendorId = 1, $disabled = [])
    {
        if (!VirtuemartHelper::isInstalled()) {
            return [];
        }

        $cacheKey = 'vmCategory.options.parentId' . $parentId . '.level:' . $level . '.vendorId:' . $vendorId . 'disabled:' . json_encode($disabled);
        $cache    = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
                        ->createCacheController('output', ['defaultgroup' => 'plg_system_vpprime']);

        if ($cache->contains($cacheKey)) {
            $options = $cache->get($cacheKey);
        } else {
            $options = self::getCategoryOptions($parentId, $level, $vendorId, $disabled);

            $cache->store($options, $cacheKey);
        }

        return $options;
    }

    protected static function getCategoryOptions($parentId = 0, $level = 0, $vendorId = 1, $disabled = [], $reset = true)
    {
        if ($reset) {
            self::$categoryOptions = [];
        }

        $level++;

        $categoryModel = \VmModel::getModel('category');

        $categoryModel->_noLimit = true;

        $categories = $categoryModel->getCategories(true, $parentId, false, '', $vendorId);

        if (!empty($categories)) {
            foreach ($categories as $category) {
                $isDisabled   = in_array($category->virtuemart_category_id, $disabled);
                $categoryId   = $category->virtuemart_category_id;
                $categoryName = str_repeat(' - ', ($level - 1)) . $category->category_name;

                self::$categoryOptions[] = HTMLHelper::_('select.option', $categoryId, $categoryName, 'value', 'text', $isDisabled);

                if ($categoryModel->hasChildren($categoryId)) {
                    self::getCategoryOptions($categoryId, $level, $vendorId, $disabled, false);
                }
            }
        }

        return self::$categoryOptions;
    }

    /**
     * Generates an HTML radio list.
     *
     * @param   array    $data       An array of objects
     * @param   string   $name       The value of the HTML name attribute
     * @param   string   $attribs    Additional HTML attributes for the `<select>` tag
     * @param   mixed    $optKey     The key that is selected
     * @param   string   $optText    The name of the object variable for the option value
     * @param   string   $selected   The name of the object variable for the option text
     * @param   boolean  $idtag      Value of the field id or null by default
     * @param   boolean  $translate  True if options will be translated
     *
     * @return  string  HTML for the select list
     *
     * @since   1.5
     */
    public static function radiolist(
        $data,
        $name,
        $attribs = null,
        $optKey = 'value',
        $optText = 'text',
        $selected = null,
        $idtag = false,
        $translate = false
    ) {
        // Set default options
        $options = array_merge(HTMLHelper::$formatOptions, ['format.depth' => 0, 'id' => false]);

        if (is_array($attribs) && func_num_args() === 3) {
            // Assume we have an options array
            $options = array_merge($options, $attribs);
        } else {
            // Get options from the parameters
            $options['id']             = $idtag;
            $options['list.attr']      = $attribs;
            $options['list.translate'] = $translate;
            $options['option.key']     = $optKey;
            $options['option.text']    = $optText;
            $options['list.select']    = $selected;
        }

        if (isset($options['id'])) {
            $idtag = $options['id'];
        }

        $attribs = '';

        if (isset($options['list.attr'])) {
            if (is_array($options['list.attr'])) {
                if (isset($options['list.attr']['class'])) {
                    $options['list.attr']['class'] .= ' form-check-input';
                } else {
                    $options['list.attr']['class'] = 'form-check-input';
                }

                $attribs = ArrayHelper::toString($options['list.attr']);
            } else {
                if (strpos($options['list.attr'], 'class="') !== false) {
                    $options['list.attr'] = str_replace('class="', 'class="form-check-input ', $options['list.attr']);
                } else {
                    $options['list.attr'] = 'class="form-check-input"';
                }

                $attribs = $options['list.attr'];
            }

            if ($attribs !== '') {
                $attribs = ' ' . $attribs;
            }
        }

        if (isset($options['list.translate'])) {
            $translate = $options['list.translate'];
        }

        if (isset($options['option.key'])) {
            $optKey = $options['option.key'];
        }

        if (isset($options['option.text'])) {
            $optText = $options['option.text'];
        }

        if (isset($options['list.select'])) {
            $selected = $options['list.select'];
        }

        $id_text = $idtag ?: $name;

        $html = '<div class="controls vm-field-controls">';

        foreach ($data as $obj) {
            $k  = $obj->$optKey;
            $t  = $translate ? Text::_($obj->$optText) : $obj->$optText;
            $id = (isset($obj->id) ? $obj->id : null);

            $extra = '';
            $id    = $id ? $obj->id : $id_text . $k;

            if (is_array($selected)) {
                foreach ($selected as $val) {
                    $k2 = is_object($val) ? $val->$optKey : $val;

                    if ($k == $k2) {
                        $extra .= ' selected="selected" ';
                        break;
                    }
                }
            } else {
                $extra .= ((string) $k === (string) $selected ? ' checked="checked" ' : '');
            }

            $price      = '';
            $className  = 'form-check form-check-inline';
            $isColorBtn = false;

            if (property_exists($obj, 'customProp')) {
                if (isset($obj->customProp['price'])) {
                    $price = $obj->customProp['price'];
                }

                if (isset($obj->customProp['className'])) {
                    $className .= ' ' . trim($obj->customProp['className']);
                }

                $isColorBtn = $obj->customProp['isColorBtn'];
            }

            $html .= '<div class="' . $className . '">';
            $html .= '<input type="radio" name="' . $name . '" id="' . $id . '" value="' . $k . '" '
                    . $extra . $attribs . '>';
            $html .= '<label for="' . $id . '" class="form-check-label" id="' . $id . '-lbl">';

            if ($isColorBtn) {
                $html .= '<span class="form-check-label-text is-color-btn has-tooltip" title="' . htmlspecialchars($t, ENT_QUOTES, 'UTF-8') . '"><span class="sr-only">' . $t . '</span></span>';
            } else {
                $html .= '<span class="form-check-label-text">' . $t . '</span>';
            }


            if ($price) {
                $html .= '<span class="form-check-label-price">' . $price . '</span>';
            }

            $html .= '</label>';
            $html .= '</div>';
        }

        $html .= '</div>';

        return $html;
    }

    public static function password($name, $value = '', $attributes = [])
    {
        $defaultAttributes = [
            'name'          => $name,
            'id'            => $name,
            'value'         => $value,
            'meter'         => false,
            'class'         => '',
            'forcePassword' => false,
            'lock'          => false,
            'disabled'      => false,
            'readonly'      => false,
            'required'      => false,
            'hint'          => '',
            'rules'         => false,
            'description'   => '',
            'autocomplete'  => 'new-password',
            'size'          => null,
            'maxLength'     => 99,
            'threshold'     => 66,
            'required'      => false,
            'autofocus'     => false,
            'dataAttribute' => '',
            'minLength'     => (int) ComponentHelper::getParams('com_users')->get('minimum_length', 12),
            'minIntegers'   => (int) ComponentHelper::getParams('com_users')->get('minimum_integers', 0),
            'minSymbols'    => (int) ComponentHelper::getParams('com_users')->get('minimum_symbols', 0),
            'minUppercase'  => (int) ComponentHelper::getParams('com_users')->get('minimum_uppercase', 0),
            'minLowercase'  => (int) ComponentHelper::getParams('com_users')->get('minimum_lowercase', 0)
        ];

        $attributes = array_replace($defaultAttributes, $attributes);

        return LayoutHelper::render('joomla.form.field.password', $attributes);
    }
}
