<?php

/**
 * @package     VP Prime Framework
 *
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2012-2025 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Virtueplanet\Plugin\System\Prime\Object;

use Joomla\CMS\Helper\ModuleHelper;
use Joomla\Registry\Registry;
use stdClass;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * ModuleMenu class of VP Prime Framework.
 */
class ModuleMenu extends stdClass
{
    /**
     * Dummy Primary key to mimic Joomla menu
     *
     * @var    string
     */
    public $id;

    /**
     * Primary key
     *
     * @var    integer
     */
    public $module_id;

    /**
     * The type of menu this item belongs to
     *
     * @var    integer
     */
    public $menutype;

    /**
     * The display title of the module item
     *
     * @var    string
     */
    public $title;

    /**
     * The name of the module.
     *
     * @var    string
     */
    public $module;

    /**
     * The module object.
     *
     * @var    object
     */
    public $moduleObject;

    /**
     * The position of the module.
     *
     * @var    string
     */
    public $position;

    /**
     * The content of the module.
     *
     * @var    string
     */
    public $content;

    /**
     * Show title of the module.
     *
     * @var    boolean
     */
    public $showtitle;

    /**
     * Menu ID of the module
     *
     * @var    integer
     */
    public $menuid;

    /**
     * Logical name of the module
     *
     * @var    string
     */
    public $name;

    /**
     * Style of the module renderer.
     *
     * @var    string
     */
    public $style;

    /**
     * If the content is rendered.
     *
     * @var    boolean
     */
    public $contentRendered;

    /**
     * The SEF alias of the module item
     *
     * @var    string
     */
    public $alias;

    /**
     * The SEF alias of the module item
     *
     * @var    string
     */
    public $slug;

    /**
     * A note associated with the module item
     *
     * @var    string
     */
    public $note;

    /**
     * The computed path of the module item based on the alias field, this is populated from the `path` field in the `#__menu` table
     *
     * @var    string
     */
    public $route;

    /**
     * The actual link the module item refers to
     *
     * @var    string
     */
    public $link;

    /**
     * Rendered HTML of the module.
     *
     * @var    string
     */
    public $html;

    /**
     * The type of link
     *
     * @var    string
     */
    public $type = 'module';

    /**
     * The relative level in the tree
     *
     * @var    integer
     */
    public $level;

    /**
     * The assigned language for this item
     *
     * @var    string
     */
    public $language = '*';

    /**
     * The click behaviour of the link
     *
     * @var    integer
     */
    public $browserNav;

    /**
     * The access level required to view the module item
     *
     * @var    integer
     */
    public $access;

    /**
     * The module item parameters
     *
     * @var    string|Registry
     * @note   This field is public as it is accessed module helper to render the module.
     */
    public $params;

    /**
     * Indicates if this module item is the home or default page
     *
     * @var    integer
     */
    public $home;

    /**
     * The image of the module item
     *
     * @var    string
     */
    public $img;

    /**
     * The optional template style applied to this module item
     *
     * @var    integer
     */
    public $template_style_id;

    /**
     * The extension ID of the component this module item is for
     *
     * @var    integer
     */
    public $component_id;

    /**
     * The parent module item in the menu tree
     *
     * @var    string
     */
    public $parent_id;

    /**
     * Active category flag.
     *
     * @var    boolean
     */
    public $active;

    /**
     * The name of the component this module item is for
     *
     * @var    string
     */
    public $component = 'com_modules';

    /**
     * The tree of parent module items
     *
     * @var    array
     */
    public $tree = [];

    /**
     * An array of the query string values for this item
     *
     * @var    array
     */
    public $query = [];

    /**
     * Parent module item/menu item
     *
     * @var   Joomla\CMS\Menu\MenuItem
     */
    protected $_parent;

    /**
     * An array of active category ids.
     *
     * @var    array
     */
    protected static $activeCategories = null;

    /**
     * Class constructor
     *
     * @param   object   $data  The module item data to load
     * @param   mixed    $calledMenuItem
     */
    public function __construct($data, $calledMenuItem = null)
    {
        foreach ($data as $key => $value) {
            $this->$key = $value;
        }

        $this->moduleObject = $data;
        $this->module_id    = $this->id;
        $this->id           = 'module-' . $this->module_id;
        $this->active       = false;
        $this->parent_id    = 0;
        $this->level        = 1;
        $this->link         = '';

        if ($calledMenuItem !== null) {
            $this->level     = $calledMenuItem->level + 1;
            $this->menutype  = $calledMenuItem->menutype;
            $this->parent_id = $calledMenuItem->id;
        }
    }

    /**
     * Returns the module item parameters
     *
     * @return  Registry
     */
    public function getParams()
    {
        if (!($this->params instanceof Registry)) {
            try {
                $this->params = new Registry($this->params);
            } catch (\RuntimeException $e) {
                /*
                 * Joomla shipped with a broken sample json string for 4 years which caused fatals with new
                 * error checks. So for now we catch the exception here - but one day we should remove it and require
                 * valid JSON.
                 */
                $this->params = new Registry();
            }
        }

        return $this->params;
    }

    /**
     * Sets the module item parameters
     *
     * @param   Registry|string  $params  The data to be stored as the parameters
     *
     * @return  void
     */
    public function setParams($params)
    {
        $this->params = $params;
    }

    /**
     * Set parent item
     *
     * @return   void
     */
    public function setParent($parent)
    {
        if ($parent instanceof \Joomla\CMS\Menu\MenuItem) {
            $this->_parent = $parent;
        }
    }

    /**
     * Get parent item
     *
     * @return   Joomla\CMS\Menu\MenuItem
     */
    public function getParent()
    {
        return $this->_parent;
    }

    /**
     * Build tree
     *
     * @return   void
     */
    public function buildTree()
    {
        $this->tree = [$this->id];

        if ($this->_parent) {
            $this->tree = array_merge($this->tree, $this->_parent->tree);
        }
    }

    /**
     * Render module HTML
     *
     * @return   void
     */
    public function render($force = false)
    {
        if ($force || !$this->html) {
            $this->html = ModuleHelper::renderModule($this->moduleObject, ['style' => 'megmenu']);
        }
    }
}
