<?php

/**
 * @package     VP Prime Framework
 *
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2012-2025 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Virtueplanet\Plugin\System\Prime\Helper;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Cache\Controller\OutputController;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Global Helper class of VP Prime Framework.
 */
class GlobalHelper
{
    /**
     * List of data
     *
     * @var   array
     */
    protected $data = [];

    /**
     * Cache key
     *
     * @var   string
     */
    protected $cacheKey = null;

    /**
     * Flag if data is stored
     *
     * @var   boolean
     */
    protected $stored = false;

    /**
     * If the data were loaded from cache
     *
     * @var   boolean
     */
    protected $hadCache = false;

    /**
     * Instance object
     *
     * @var   null|\Virtueplanet\Plugin\System\Prime\Helper\GlobalHelper
     */
    protected static $instance = null;

    /**
     * Constructor function
     *
     * @return   void
     */
    public function __construct()
    {
        // Load data from cache
        // when the instance is created.
        $this->loadDataFromCache();
    }

    /**
     * Get an instance of the GlobalHelper class
     *
     * @return    \Virtueplanet\Plugin\System\Prime\Helper\GlobalHelper
     */
    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Set data by the key name.
     *
     * @param    string   $name   The key of the data.
     * @param    mixed    $value  The data to be stored.
     *
     * @return   void
     */
    public function setData(string $name, $value)
    {
        $this->data[$name] = $value;
    }

    /**
     * Get data by the key name.
     *
     * @param    string   $name     The key of the data.
     * @param    mixed    $default  The default value in case key not found.
     *
     * @return   mixed
     */
    public function getData(string $name, $default = null)
    {
        if (isset($this->data[$name])) {
            return $this->data[$name];
        }

        return $default;
    }

    /**
     * Remove data by the key name.
     *
     * @param    string   $name   The key of the data.
     *
     * @return   void
     */
    public function removeData(string $name)
    {
        if (isset($this->data[$name])) {
            unset($this->data[$name]);
        }
    }

    /**
     * Store all data to cache.
     * Call storeToCache function after everything is printed.
     * We need to call this function in VP Prime Framework onAfterRender.
     *
     * @return   void
     */
    public function storeDataToCache()
    {
        if (!$this->stored) {
            if (!empty($this->data) || $this->hadCache) {
                /** @var OutputController $cache */
                $cache    = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
                                ->createCacheController('output', ['defaultgroup' => 'plg_system_vpprime']);
                $cacheKey = $this->getCacheKey();

                $cache->store($this->data, $cacheKey);
            }

            $this->stored = true;
        }
    }

    /**
     * Load data from cache.a11y_contrast
     *
     * @return   void
     */
    protected function loadDataFromCache()
    {
        /** @var OutputController $cache */
        $cache    = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
                        ->createCacheController('output', ['defaultgroup' => 'plg_system_vpprime']);
        $cacheKey = $this->getCacheKey();

        if ($cache->contains($cacheKey)) {
            $this->data     = (array) $cache->get($cacheKey);
            $this->hadCache = true;
        } else {
            $this->data = [];
        }
    }

    /**
     * Get cache key
     *
     * @return   string
     */
    protected function getCacheKey()
    {
        if ($this->cacheKey === null) {
            $this->cacheKey = 'plg_system_vpprime.global.' . Uri::getInstance()->toString();
        }

        return $this->cacheKey;
    }
}
