<?php
/**
 * @package    Essentials for VirtueMart
 * @author     Abhishek Das <info@virtueplanet.com>
 * @copyright  Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://www.virtueplanet.com
 */

defined ('_JEXEC') or die;

/**
* VirtueMart Essentials System Plugin
* 
* @package   VirtueMart Essentials
* @since     1.0
*/
class plgSystemVMEssentials extends JPlugin
{
	protected $app;
	
	protected $cparams;
	
	/**
	 * Constructor.
	 *
	 * @param   object  &$subject  The object to observe
	 * @param   array   $config    An optional associative array of configuration settings.
	 *
	 * @since   1.0.0
	 */
	public function __construct(&$subject, $config) 
	{
		parent::__construct($subject, $config);
		
		$this->cparams = JComponentHelper::getParams('com_vmessentials');
	}

	/**
	 * After route.
	 *
	 * @return  void
	 *
	 * @since   1.6
	 */
	public function onAfterRoute()
	{
		if (!$this->app->isClient('site'))
		{
			return;
		}
		
		if (!(@include_once(JPATH_ADMINISTRATOR . '/components/com_vmessentials/include.php')))
		{
			return;
		}
		
		if (!defined ('VMESSENTIALS_ADMINISTRATOR') || !class_exists('VmConfig'))
		{
			// VirtueMart Essentials Package not installed correctly or VirtueMart is not installed.
			return;
		}
		
		VmConfig::loadConfig();

		// Register VM Essentials helper
		JLoader::register('VMEssentialsHelpersVMEssentials', VMESSENTIALS_SITE . '/helpers/vmessentials.php');
		
		// Load component language
		JFactory::getLanguage()->load('com_vmessentials');
		
		// Register html path
		JHtml::addIncludePath(VMESSENTIALS_SITE . '/helpers/html');
		
		// Move guest lists to user account after login
		VMEssentialsHelpersVMEssentials::moveWishlistToUser();
	}
	
	/**
	 * After route.
	 *
	 * @return  void
	 *
	 * @since   1.6
	 */
	public function onAfterDispatch()
	{
		if (!$this->app->isClient('site'))
		{
			return;
		}
		
		if (!(@include_once(JPATH_ADMINISTRATOR . '/components/com_vmessentials/include.php')))
		{
			return;
		}
		
		if (!defined ('VMESSENTIALS_ADMINISTRATOR') || !class_exists('VmConfig'))
		{
			// VirtueMart Essentials Package not installed correctly or VirtueMart is not installed.
			return;
		}
		
		if ($this->_isHTML())
		{
			// Load required assets
			JHtml::_('vmessentials.assets');
		}
	}

	/**
	 * After render.
	 *
	 * @return  void
	 *
	 * @since   1.6
	 */
	public function onAfterRender()
	{
		if (!$this->app->isClient('site'))
		{
			return;
		}
		
		if (!defined ('VMESSENTIALS_ADMINISTRATOR') || !class_exists('VmConfig'))
		{
			// VirtueMart Essentials Package not installed correctly or VirtueMart is not installed.
			return;
		}
		
		if ($this->_isHTML() && $this->cparams->get('auto_buttons', 1))
		{
			// Register parser helper
			JLoader::register('VMEssentialsParser', VMESSENTIALS_SITE . '/helpers/parser.php');
			
			// Create an instance of parser helper
			$parser = new VMEssentialsParser();
			
			// Parse and add buttons to products
			$parser->addButtons();
		}
	}
	
	/**
	 * On Ajax call.
	 *
	 * @return  void
	 *
	 * @since   1.6
	 */
	public function onAjaxVmessentials()
	{
		if ($this->app->isClient('site'))
		{
			return;
		}
		
		if (!(@include_once(JPATH_ADMINISTRATOR . '/components/com_vmessentials/include.php')))
		{
			return;
		}
		
		// Register dlk helper class
		JLoader::register('VPComDlkHelper', VMESSENTIALS_ADMINISTRATOR . '/models/fields/vpcomdlk/helper.php');
		
		// Get an instance of dlk helper
		$dlkHelper = VPComDlkHelper::getInstance();
		
		if ($this->app->input->getCmd('method') == 'validatedlk')
		{
			// Return validation result
			return $dlkHelper->validate();
		}
		elseif ($this->app->input->getCmd('method') == 'revalidatedlk')
		{
			// Return validation result
			return $dlkHelper->revalidate();
		}
	}

	/**
	 * Called on live update installation.
	 *
	 * @return  boolean
	 *
	 * @since   1.0
	 */
	public function onInstallerBeforePackageDownload(&$url, &$headers)
	{
		if (!(@include_once(JPATH_ADMINISTRATOR . '/components/com_vmessentials/include.php')))
		{
			return;
		}
		
		// Register dlk helper class
		JLoader::register('VPComDlkHelper', VMESSENTIALS_ADMINISTRATOR . '/models/fields/vpcomdlk/helper.php');
		
		return VPComDlkHelper::getInstance()->addDlk($url, $headers);
	}
	
	/**
	* Private method to check if the loaded document is HTML
	* 
	* @return boolean (true/false)
	*/
	private function _isHTML()
	{
		$doc   = JFactory::getDocument();
		$input = $this->app->input;
		
		$isFeed = ($doc->getType() == 'feed' || $input->getWord('format') == 'feed' || $input->getWord('type') == 'rss' || $input->getWord('type') == 'atom');
		$isHTML = ($doc->getType() == 'html' && !$isFeed);
		
		return $isHTML;
	}
}
