<?php

/**
 * @package     VP Prime Framework
 *
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2023-2025 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Virtueplanet\Module\Manufacturer\Site\Helper;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Cache\Controller\OutputController;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Router\Route;
use Joomla\Database\DatabaseInterface;
use Virtueplanet\Plugin\System\Prime\Helper\VirtuemartHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Helper for mod_vpprime_mega_menu
 */
class ManufacturerHelper
{
    /**
     * Get a list of the menu items.
     *
     * @param   \Joomla\Registry\Registry  &$params  The module options.
     *
     * @return  array
     */
    public static function getList(&$params)
    {
        /** @var \Joomla\CMS\Application\SiteApplication $app */
        $app  = Factory::getApplication();

        // Compose cache key
        $cacheKey = 'manufacturers.' . $params;

        /** @var OutputController $cache */
        $cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
            ->createCacheController('output', ['defaultgroup' => 'mod_vpprime_manufacturer_carousel']);

        if ($cache->contains($cacheKey)) {
            $items = $cache->get($cacheKey);
        } else {
            $display       = $params->get('display', 'all');
            $categories    = (array) $params->get('categories', []);
            $manufacturers = (array) $params->get('manufacturers', []);

            switch ($display) {
                case 'by_categories':
                    $items = self::getManufacturers($categories);
                    break;
                case 'selected':
                    $items = self::getManufacturers([], $manufacturers);
                    break;
                default:
                case 'all':
                    $items = self::getManufacturers();
                    break;
            }

            $cache->store($items, $cacheKey);
        }

        return $items;
    }

    protected static function getManufacturers($categoryIds = [], $manufacturerIds = [])
    {
        \JLoader::register('VmConfig', JPATH_ADMINISTRATOR . '/components/com_virtuemart/helpers/config.php');

        \VmConfig::loadConfig();

        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);

        $query->select($db->quoteName(['m.virtuemart_manufacturer_id', 'm.virtuemart_manufacturercategories_id']))
            ->from($db->quoteName('#__virtuemart_manufacturers') . ' AS ' . $db->quoteName('m'))
            ->select($db->quoteName(['ml.mf_name', 'mf_email', 'mf_desc', 'mf_url']))
            ->join('INNER', $db->quoteName('#__virtuemart_manufacturers_' . \VmConfig::$vmlang) . ' AS ' . $db->quoteName('ml') . ' ON ' . $db->quoteName('ml.virtuemart_manufacturer_id') . ' = ' . $db->quoteName('m.virtuemart_manufacturer_id'))
            ->select($db->quoteName('mc.mf_category_name'))
            ->join('LEFT', $db->quoteName('#__virtuemart_manufacturercategories_' . \VmConfig::$vmlang) . ' AS ' . $db->quoteName('mc') . ' ON ' . $db->quoteName('mc.virtuemart_manufacturercategories_id') . ' = ' . $db->quoteName('m.virtuemart_manufacturercategories_id'))
            ->select($db->quoteName('mm.virtuemart_media_id'))
            ->join('LEFT', $db->quoteName('#__virtuemart_manufacturer_medias') . ' AS ' . $db->quoteName('mm') . ' ON ' . $db->quoteName('mm.virtuemart_manufacturer_id') . ' = ' . $db->quoteName('m.virtuemart_manufacturer_id'))
            ->where($db->quoteName('m.published') . ' = 1')
            ->order($db->quoteName('ml.mf_name') . ' ASC')
            ->group($db->quoteName('m.virtuemart_manufacturer_id'));

        if (!empty($categoryIds)) {
            if (count($categoryIds) === 1) {
                $query->where($db->quoteName('m.virtuemart_manufacturercategories_id') . ' = ' . reset($categoryIds));
            } else {
                $query->where($db->quoteName('m.virtuemart_manufacturercategories_id') . ' IN (' . implode(',', $categoryIds) . ')');
            }
        }

        if (!empty($manufacturerIds)) {
            if (count($manufacturerIds) === 1) {
                $query->where($db->quoteName('m.virtuemart_manufacturer_id') . ' = ' . reset($manufacturerIds));
            } else {
                $query->where($db->quoteName('m.virtuemart_manufacturer_id') . ' IN (' . implode(',', $manufacturerIds) . ')');
            }
        }

        $db->setQuery($query);

        $manufactuers = $db->loadObjectList();

        if (!empty($manufactuers)) {
            /** @var \VirtueMartModelManufacturer  $model */
            $model = \VmModel::getModel('Manufacturer');

            $model->addImages($manufactuers, 1);
        }

        return $manufactuers;
    }
}
