<?php

/**
 * @package     VP Prime Ajax Product Search
 *
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2012-2023 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Virtueplanet\Module\AjaxSearch\Site\Helper;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Cache\Controller\OutputController;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Router\SiteRouter;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseInterface;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Module Helper class of VP Prime Framework.
 */
abstract class SearchHelper
{
    protected static $modules = [];

    /**
     * Display the search button as an image.
     *
     * @param   string  $button_text  The alt text for the button.
     *
     * @return  string  The HTML for the image.
     */
    public static function getValidNumber($params, $param, $default = 0)
    {
        $postiveInteger = (int) $params->get($param, 0);

        if ($postiveInteger <= 0) {
            return $default;
        }

        return $postiveInteger;
    }

    public static function getWidthPerRow($params, $param, $default = 0)
    {
        $per_row = self::getValidNumber($params, $param, $default);
        $width   = 100;

        if ($per_row > 0) {
            $width = (100 / $per_row);

            if ($width != round($width)) {
                $width = number_format($width, 8, '.', '');
            }
        }

        return (string) $width;
    }

    public static function getCategoryId($params)
    {
        $categoryId = 0;

        if ($params->get('by_category', 0)) {
            $input  = Factory::getApplication()->input;
            $option = strtolower($input->getCmd('option', ''));
            $view   = strtolower($input->getCmd('view', ''));
            $catId  = $input->getInt('virtuemart_category_id', 0);

            if ($option == 'com_virtuemart' && ($view == 'category' || $view == 'productdetails') && $catId > 0) {
                $categoryId = (int) $catId;
            }
        }

        return $categoryId;
    }

    public static function getHiddenFields($params, $actionUrl)
    {
        /** @var \Joomla\CMS\Application\SiteApplication $app */
        $app    = Factory::getApplication();
        $uri    = clone Uri::getInstance($actionUrl);
        $catId  = self::getCategoryId($params);
        $itemid = (int) $params->get('set_itemid', 0);
        $fields = [];
        $vars   = [];

        if ($app->get('force_ssl') === 2 && (!$uri->getScheme() || strtolower($uri->getScheme()) !== 'https')) {
            $uri->setScheme('https');
        }

        if ($app->get('sef') && $params->get('clean_sef_search_url', 1)) {
            $router = Factory::getContainer()->get(SiteRouter::class);
            $menu   = $app->getMenu();

            // Look for the home menu
            if (Multilanguage::isEnabled()) {
                $defaultMenu = $menu->getDefault(Factory::getLanguage()->getTag());
            } else {
                $defaultMenu = $menu->getDefault();
            }

            $activeMenu = $menu->getActive() ?: $defaultMenu;

            try {
                $vars = $router->parse($uri, true);
            } catch (\Exception $e) {
                // We do not need to handle the exception here.
            }

            // Joomla\CMS\Router\SiteRouter sets active menu after parsing.
            // So we need to reset it.
            if ($activeMenu) {
                $menu->setActive($activeMenu->id);
            }
        } else {
            $vars = $uri->getQuery(true);
        }

        if (!isset($vars['option']) || $vars['option'] != 'com_virtuemart') {
            $fields['option'] = 'com_virtuemart';
        }

        if (!isset($vars['view']) || $vars['view'] != 'category') {
            $fields['view'] = 'category';
        }

        if (!isset($vars['limitstart']) || $vars['limitstart'] != 0) {
            $fields['limitstart'] = 0;
        }

        if (!$params->get('show_category_selector', 1)) {
            if (!isset($vars['virtuemart_category_id']) || $vars['virtuemart_category_id'] != $catId) {
                $fields['virtuemart_category_id'] = (int) $catId;
            }
        }

        if ($itemid > 0 && (!isset($vars['Itemid']) || $vars['Itemid'] != $itemid)) {
            $fields['Itemid'] = (int) $itemid;
        }

        return $fields;
    }

    public static function getCategoryListOptions($parentId = 0, $level = 0, $vendorId = 1, $disabled = [])
    {
        $cacheKey = 'virtuemart.category.options.parentId:' . $parentId . '.level:' . $level . '.vendorId:' . $vendorId . '.disabled:' . json_encode($disabled);

        /** @var OutputController $cache */
        $cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
            ->createCacheController('output', ['defaultgroup' => 'mod_vpprime_ajax_search']);

        if ($cache->contains($cacheKey)) {
            $options = $cache->get($cacheKey);
        } else {
            $options = self::buildCategoryListOptions($parentId, $level, $vendorId, $disabled);

            $cache->store($options, $cacheKey);
        }

        return $options;
    }

    protected static function buildCategoryListOptions($parentId = 0, $level = 0, $vendorId = 1, $disabled = [])
    {
        static $options = [];

        $categoryModel = \VmModel::getModel('category');

        $categoryModel->_noLimit = true;

        $categories = $categoryModel->getCategories(true, $parentId, false, '', $vendorId);

        $level++;

        if (!empty($categories)) {
            foreach ($categories as $category) {
                $isDisabled   = in_array($category->virtuemart_category_id, $disabled);
                $categoryId   = $category->virtuemart_category_id;
                $categoryName = str_repeat(' - ', ($level - 1)) . $category->category_name;

                $options[] = HTMLHelper::_('select.option', $categoryId, $categoryName, 'value', 'text', $isDisabled);

                if ($categoryModel->hasChildren($categoryId)) {
                    self::buildCategoryListOptions($categoryId, $level, $vendorId, $disabled);
                }
            }
        }

        return $options;
    }

    public static function getCategoryNames($categoryIds)
    {
        $model = \VmModel::getModel('category');
        $names = [];

        foreach ($categoryIds as $categoryId) {
            $category = $model->getCategory($categoryId, false, true);

            if (!empty($category->category_name)) {
                $names[] = $category->category_name;
            }
        }

        return $names;
    }

    public static function getModuleById($moduleId)
    {
        $moduleId = (int) $moduleId;

        if ($moduleId <= 0) {
            return false;
        }

        if (!isset(self::$modules[$moduleId])) {
            $db    = Factory::getContainer()->get(DatabaseInterface::class);
            $query = $db->getQuery(true);
            $user  = Factory::getApplication()->getIdentity();

            $query->select($db->quoteName(['id', 'title', 'module', 'position', 'content', 'showtitle', 'params']))
                ->from($db->quoteName('#__modules'))
                ->where($db->quoteName('id') . ' = ' . $moduleId)
                ->where($db->quoteName('published') . ' = 1')
                ->where($db->quoteName('access') . ' IN (' . implode(',', $db->quote($user->getAuthorisedViewLevels())) . ')');

            $db->setQuery($query);
            self::$modules[$moduleId] = $db->loadObject();
        }

        return self::$modules[$moduleId];
    }
}
