/*
 * @package     VP Prime Framework
 * @author      Abhishek Das <info@virtueplanet.com>
 * @link        https://www.virtueplanet.com
 * @copyright   Copyright (C) 2023-2025 Virtueplanet Services LLP. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

var VPPrimeAdmin = window.VPPrimeAdmin || {};

(function (app, document, window) {

    app.primeContainer = null;

    app.dlkContainer = null;

    app.dlkForm = null;

    app.init = function () {
        [].slice.call(document.querySelectorAll('.options-form')).forEach(function (fieldset) {
            if (fieldset.parentElement && fieldset.parentElement.tagName === 'JOOMLA-TAB-ELEMENT' && !fieldset.parentElement.classList.contains('vp-admin-group') && fieldset.parentElement.querySelectorAll('.options-form').length > 1) {
                fieldset.parentElement.classList.add('vp-admin-group');
            }
        });

        [].slice.call(document.querySelectorAll('.primeadmin')).forEach(function (adminField) {
            adminField.closest('.control-group').classList.add('primeadmin-group');

            var revalidateDlkField = adminField.querySelector('input[name="revalidate_dlk"]');

            if (revalidateDlkField) {
                app.dlkContainer.setAttribute('data-revalidate', revalidateDlkField.value);
                app.dlkContainer.dataset.revalidate = revalidateDlkField.value;
            }
        });

        // Move Dlk modal form 
        var dlkModal = document.getElementById('dlkFormModal');

        if (dlkModal) {
            document.body.appendChild(dlkModal);
        }

        if (app.dlkForm) {
            app.bindDlkForm();
        }

        if (app.dlkContainer) {
            app.bindRevalidateDlkBtn();
            app.bindRemoveDlkBtn();

            if (app.dlkContainer.getAttribute('data-revalidate') == 1 && app.dlkContainer.querySelector('[data-dlk="revalidate"]')) {
                app.dlkContainer.querySelector('[data-dlk="revalidate"]').click();
            }
        }
    }

    app.bindDlkForm = function () {
        var unameField = app.dlkForm.querySelector('[data-dlkform="uname"]');
        var dlkField = app.dlkForm.querySelector('[data-dlkform="dlk"]');
        var submit = app.dlkForm.querySelector('[data-dlkform="submit"]');

        submit.addEventListener('click', function (event) {
            event.preventDefault();

            if (!unameField.value) {
                app.showDlkMessage(Joomla.Text._('PLG_SYSTEM_VPPRIME_PRIMEADMIN_FIELD_ERROR_UNAME'), 'danger');

                return false;
            }

            if (!dlkField.value) {
                app.showDlkMessage(Joomla.Text._('PLG_SYSTEM_VPPRIME_PRIMEADMIN_FIELD_ERROR_DLK'), 'danger');

                return false;
            }

            submit.setAttribute('disabled', true);

            app.post({
                method: 'validateDlk',
                uname: unameField.value,
                dlk: dlkField.value,
                template_style_id: app.dlkContainer.dataset.tmplstyleid
            }).then(function (result) {
                if (result.error) {
                    var isWarning = false;

                    if (typeof result.data !== 'undefined' && result.data.is_warning) {
                        isWarning = true;
                    }

                    app.showDlkMessage(result.message, (isWarning ? 'warning' : 'danger'));

                    if (!isWarning) {
                        submit.removeAttribute('disabled');
                        return false;
                    }
                }

                if (result.data.dlk && result.data.access_code > 0) {
                    app.dlkContainer.setAttribute('data-hasdlk', 1);
                    app.dlkContainer.setAttribute('data-accesscode', result.data.access_code);
                    app.dlkContainer.querySelector('[data-dlkfield]').value = result.data.dlk;

                    if (result.data.last_checked) {
                        app.dlkContainer.querySelector('[data-lastchecked]').innerHTML = result.data.last_checked;
                    }

                    if (app.primeContainer) {
                        var updateBtn = app.primeContainer.querySelector('[data-update="btn"]');

                        if (updateBtn) {
                            updateBtn.removeAttribute('disabled');
                        }
                    }
                }

                if (result.message) {
                    app.showDlkMessage(result.message, 'success');
                    app.hideModalForm();
                }

                submit.removeAttribute('disabled');
            }).catch(function (error) {
                console.error(error);
            });
        })
    }

    app.bindRevalidateDlkBtn = function () {
        var btn = app.dlkContainer.querySelector('[data-dlk="revalidate"]');

        if (!btn) {
            return false;
        }

        btn.addEventListener('click', function (event) {
            event.preventDefault();

            btn.setAttribute('disabled', true);

            app.post({
                method: 'revalidateDlk',
                template_style_id: app.dlkContainer.dataset.tmplstyleid
            }).then(function (result) {
                if (result.error) {
                    var isWarning = false;

                    if (typeof result.data !== 'undefined' && result.data.is_warning) {
                        isWarning = true;
                    }

                    app.showSystemMessage(result.message, (isWarning ? 'warning' : 'danger'));

                    if (!isWarning) {
                        btn.removeAttribute('disabled');
                        return false;
                    }
                }

                if (result.data.dlk && result.data.access_code > 0) {
                    app.dlkContainer.setAttribute('data-hasdlk', 1);
                    app.dlkContainer.setAttribute('data-accesscode', result.data.access_code);
                    app.dlkContainer.querySelector('[data-dlkfield]').value = result.data.dlk;

                    if (result.data.last_checked) {
                        app.dlkContainer.querySelector('[data-lastchecked]').innerHTML = result.data.last_checked;
                    }

                    if (app.primeContainer) {
                        var updateBtn = app.primeContainer.querySelector('[data-update="btn"]');

                        if (updateBtn) {
                            updateBtn.removeAttribute('disabled');
                        }
                    }
                }

                if (result.message) {
                    app.showSystemMessage(result.message, 'success');
                }

                btn.removeAttribute('disabled');
            }).catch(function (error) {
                console.error(error);
            });
        })
    }

    app.bindRemoveDlkBtn = function () {
        var btn = app.dlkContainer.querySelector('[data-dlk="remove"]');

        if (!btn) {
            return false;
        }

        btn.addEventListener('click', function (event) {
            event.preventDefault();

            btn.setAttribute('disabled', true);

            app.post({
                method: 'removeDlk',
                template_style_id: app.dlkContainer.dataset.tmplstyleid
            }).then(function (result) {
                if (result.error) {
                    var isWarning = false;

                    if (typeof result.data !== 'undefined' && result.data.is_warning) {
                        isWarning = true;
                    }

                    app.showSystemMessage(result.message, (isWarning ? 'warning' : 'danger'));

                    if (!isWarning) {
                        btn.removeAttribute('disabled');
                        return false;
                    }
                }

                app.dlkContainer.setAttribute('data-hasdlk', 0);
                app.dlkContainer.setAttribute('data-accesscode', 0);
                app.dlkContainer.querySelector('[data-dlkfield]').value = '';

                if (app.primeContainer) {
                    var updateBtn = app.primeContainer.querySelector('[data-update="btn"]');

                    if (updateBtn) {
                        updateBtn.setAttribute('disabled', true);
                    }
                }

                if (result.message) {
                    app.showSystemMessage(result.message, 'success');
                }

                btn.removeAttribute('disabled');
            }).catch(function (error) {
                console.error(error);
            });
        })
    }

    app.showSystemMessage = function (message, type) {
        var types = ['success', 'danger', 'warning', 'info'];

        if (!type || typeof type !== 'string' || !types.includes(type)) {
            type = 'warning';
        }

        var messages = {};

        messages[type] = [message];

        Joomla.renderMessages(messages);
    }

    app.showDlkMessage = function (message, type) {
        if (!app.dlkForm) {
            return false;
        }

        var messageContainer = app.dlkForm.querySelector('[data-dlkform="message"]');

        if (messageContainer) {
            var types = ['success', 'danger', 'warning', 'info'];
            var alert = document.createElement('div');

            alert.setAttribute('role', 'alert');
            alert.classList.add('alert');

            if (type && typeof type === 'string' && types.includes(type)) {
                alert.classList.add('alert-' + type);
            } else {
                alert.classList.add('alert-warning');
            }

            alert.innerHTML = message;

            messageContainer.innerHTML = '';

            messageContainer.appendChild(alert);
        }
    }

    app.clearDlkMessage = function () {
        if (!app.dlkForm) {
            return false;
        }

        var messageContainer = app.dlkForm.querySelector('[data-dlkform="message"]');

        if (messageContainer) {
            messageContainer.innerHTML = '';
        }
    }

    app.verifyFiles = function () {
        [].slice.call(document.querySelectorAll('[data-action="verifyfiles"]')).forEach(function (element) {
            var template_style_id = !element.dataset.tmplstyleid ? 0 : element.dataset.tmplstyleid;

            app.get({
                method: 'compareFileHashes',
                template_style_id: template_style_id
            }).then(function (result) {
                if (result.error) {
                    element.setAttribute('data-state', 'error');

                    if (result.message) {
                        app.showSystemMessage(result.message, 'danger');
                    }

                    return false;
                }

                if (!result.data || typeof result.data !== 'object') {
                    element.setAttribute('data-state', 'error');

                    return false;
                }

                element.setAttribute('data-state', 'loaded');

                var data = result.data;

                if (data.changed_count > 0 || data.new_count > 0 || data.deleted_count > 0) {
                    element.setAttribute('data-found', 1);

                    if (data.changed_count > 0 && data.changed.length) {
                        element.setAttribute('data-haschange', 1);
                        element.querySelector('[data-element="changed-count"]').innerText = data.changed_count;
                        element.querySelector('[data-element="changed-files"]').innerHTML = data.changed.join("\n");
                    }

                    if (data.new_count > 0 && data.new.length) {
                        element.setAttribute('data-hasnew', 1);
                        element.querySelector('[data-element="new-count"]').innerText = data.new_count;
                        element.querySelector('[data-element="new-files"]').innerHTML = data.new.join("\n");
                    }

                    if (data.deleted_count > 0 && data.deleted.length) {
                        element.setAttribute('data-hasdeleted', 1);
                        element.querySelector('[data-element="deleted-count"]').innerText = data.deleted_count;
                        element.querySelector('[data-element="deleted-files"]').innerHTML = data.deleted.join("\n");
                    }
                }
            }).catch(function (error) {
                element.setAttribute('data-state', 'error');
                console.log(error);
            })
        })
    }

    app.getUpdate = function () {
        [].slice.call(document.querySelectorAll('[data-action="getversion"]')).forEach(function (cell) {
            var extensionId = !cell.dataset.extensionid ? 0 : cell.dataset.extensionid;
            var currentVersion = !cell.dataset.currentversion ? '' : cell.dataset.currentversion;
            var updateTable = cell.closest('table');
            var currentVersionElement = cell.closest('tr').querySelector('.current-version');
            var updateFoundCell = updateTable.querySelector('.update-found');

            if (!extensionId || !currentVersion) {
                cell.classList.add('has-error');
                return false;
            }

            app.get({
                method: 'getExtensionUpdate',
                extension_id: extensionId,
                current_version: currentVersion
            }).then(function (result) {
                if (result.error) {
                    cell.classList.add('has-error');

                    if (result.message) {
                        console.error(result.message);
                    }

                    return false;
                }

                cell.innerHTML = result.data.version;

                if (result.data.has_update && updateFoundCell) {
                    updateFoundCell.classList.remove('d-none');

                    if (result.data.infourl) {
                        var infoLink = updateFoundCell.querySelector('[data-update="infourl"]');

                        if (infoLink) {
                            infoLink.setAttribute('href', result.data.infourl);
                            infoLink.classList.remove('d-none');
                        }
                    }
                }

                if (currentVersionElement) {
                    currentVersionElement.classList.remove(currentVersionElement.dataset.dclassname);

                    if (result.data.has_update) {
                        currentVersionElement.classList.add(currentVersionElement.dataset.oclassname);
                    } else {
                        currentVersionElement.classList.add(currentVersionElement.dataset.nclassname);
                    }
                }
            }).catch(function (error) {
                cell.classList.add('has-error');
                console.log(error);
            });
        });
    }

    app.get = function (data) {
        var systemPaths = Joomla.getOptions('system.paths', {});
        var csrfToken = Joomla.getOptions('csrf.token', '');
        var base = '';

        if (typeof systemPaths.base !== typeof undefined) {
            base = systemPaths.base;
        }

        var defaultData = {};

        if (csrfToken) {
            defaultData[csrfToken] = 1;
        }

        defaultData['_' + Date.now()] = 1;

        data = Object.assign({}, data, defaultData);

        var url = base + '/index.php?option=com_ajax&plugin=vpprime&format=json&' + new URLSearchParams(data).toString();

        return new Promise(function (resolve, reject) {
            var request = new XMLHttpRequest();

            request.getResponseHeader('Content-type', 'application/json');
            request.responseType = 'json';

            request.onreadystatechange = function () {
                if (request.readyState === XMLHttpRequest.DONE) {
                    if (request.status === 200) {
                        resolve(request.response);
                    } else {
                        reject(Error(request.status));
                    }
                }
            }

            request.onerror = function () {
                reject(Error('Network Error'));
            }

            request.open('GET', url, true);

            // via Cache-Control header:
            request.setRequestHeader('Cache-Control', 'no-cache, no-store, max-age=0');

            // fallbacks for IE and older browsers:
            request.setRequestHeader('Expires', 'Tue, 01 Jan 1980 1:00:00 GMT');
            request.setRequestHeader('Pragma', 'no-cache');

            request.send();
        });
    }

    app.post = function (data) {
        var systemPaths = Joomla.getOptions('system.paths', {});
        var csrfToken = Joomla.getOptions('csrf.token', '');
        var base = '';

        if (typeof systemPaths.base !== typeof undefined) {
            base = systemPaths.base;
        }

        var defaultData = {};

        if (csrfToken) {
            defaultData[csrfToken] = 1;
        }

        data = Object.assign({}, data, defaultData);

        var url = base + '/index.php?option=com_ajax&plugin=vpprime&format=json';
        var formData = new FormData();

        Object.entries(data).forEach(([key, value]) => {
            formData.append(key, value);
        });

        return new Promise(function (resolve, reject) {
            var request = new XMLHttpRequest();

            request.getResponseHeader('Content-type', 'application/json');
            request.responseType = 'json';

            request.onreadystatechange = function () {
                if (request.readyState === XMLHttpRequest.DONE) {
                    if (request.status === 200) {
                        resolve(request.response);
                    } else {
                        reject(Error(request.status));
                    }
                }
            }

            request.onerror = function () {
                reject(Error('Network Error'));
            }

            request.open('POST', url, true);

            // via Cache-Control header:
            request.setRequestHeader('Cache-Control', 'no-cache, no-store, max-age=0');

            // fallbacks for IE and older browsers:
            request.setRequestHeader('Expires', 'Tue, 01 Jan 1980 1:00:00 GMT');
            request.setRequestHeader('Pragma', 'no-cache');

            request.send(formData);
        });
    }

    app.hideModalForm = function (timeout) {
        var dlkFormModal = document.getElementById('dlkFormModal');
        timeout = !timeout || timeout < 0 ? 1000 : timeout;

        setTimeout(function () {
            if (dlkFormModal && dlkFormModal.classList.contains('show')) {
                var modal = bootstrap.Modal.getInstance(dlkFormModal);

                if (modal) {
                    modal.hide();
                }
            }
        }, timeout);
    }

    window.addEventListener('load', function () {
        var dlkFormModal = document.getElementById('dlkFormModal');

        if (dlkFormModal) {
            var modal = bootstrap.Modal.getInstance(dlkFormModal);

            if (app.dlkContainer && modal) {
                var hasDlk = !app.dlkContainer.dataset.hasdlk ? 0 : parseInt(app.dlkContainer.dataset.hasdlk);

                if (!hasDlk) {
                    app.showDlkMessage(Joomla.Text._('PLG_SYSTEM_VPPRIME_PRIMEADMIN_FIELD_ON_LOAD_ADD_DLK_MESSAGE'), 'info');
                    modal.show();
                }
            }
        }

        app.getUpdate();
        app.verifyFiles();
    });

    document.addEventListener('DOMContentLoaded', function (event) {
        var elements = Array.from(document.querySelectorAll('.has-tooltip'));
        var options = {
            animation: true,
            container: 'body',
            delay: 0,
            html: true,
            selector: false,
            trigger: 'hover focus',
            fallbackPlacement: null,
            boundary: 'clippingParents',
            title: '',
            customClass: '',
            sanitize: true,
            sanitizeFn: null,
            popperConfig: null
        };

        if (elements.length) {
            elements.map(function (el) {
                return new window.bootstrap.Tooltip(el, options);
            });
        }

        var dlkFormModal = document.getElementById('dlkFormModal');

        if (dlkFormModal) {
            dlkFormModal.addEventListener('hidden.bs.modal', function () {
                document.querySelector('[data-dlkform="message"]').innerHTML = '';
                document.querySelector('[data-dlkform="uname"]').value = '';
                document.querySelector('[data-dlkform="dlk"]').value = '';
            });
        }

        app.primeContainer = document.querySelector('.primeadmin');
        app.dlkContainer = document.querySelector('.dlk-field-container');
        app.dlkForm = document.getElementById('dlkForm');

        app.init();
    });
})(VPPrimeAdmin, document, window);
