/**
 * @package    Essentials for VirtueMart
 * @author     Abhishek Das <info@virtueplanet.com>
 * @copyright  Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://www.virtueplanet.com
 */

var vmeIframeModal = window.vmeIframeModal || {};

(function(app, document, window) {

	if (typeof Element !== 'undefined') {
		if (!Element.prototype.matches) {
			Element.prototype.matches = Element.prototype.msMatchesSelector || Element.prototype.webkitMatchesSelector;
		}

		if (!Element.prototype.closest) {
			Element.prototype.closest = function(s) {
				var el = this;

				do {
					if (el.matches(s)) return el;
					el = el.parentElement || el.parentNode;
				} while (el !== null && el.nodeType === 1);

				return null;
			};
		}
	}

	app.selector = '[data-iframe-modal]';
	app.modal = null;
	app.backdrop = null;
	app.timeout = null;
	app.isOpen = false;

	app.show = function(link) {
		if (!app.modal) {
			var closeBtn = document.createElement('a');

			closeBtn.href = '#';
			closeBtn.classList.add('vme-modal-close');
			closeBtn.innerHTML = '&times;';

			closeBtn.addEventListener('click', function(e) {
				e.preventDefault();
				app.hide();
				return false;
			});

			var iframe = document.createElement('iframe');

			iframe.classList.add('vme-modal-iframe');
			iframe.src = 'about:blank';
			iframe.width = '100%';
			iframe.height = '100%';
			iframe.setAttribute('border', '0');

			iframe.addEventListener('load', function(event) {
				if (window.parent && typeof window.parent.vmeIframeModal === 'object') {
					window.parent.vmeIframeModal.setHeight('');
					window.parent.vmeIframeModal.setHeight(this.contentWindow.document.body.offsetHeight);
					window.parent.vmeIframeModal.stopLoading();
				}
			});

			var modalLoader = document.createElement('div');

			modalLoader.classList.add('vme-modal-loader');

			var modalBody = document.createElement('div');

			modalBody.classList.add('vme-modal-body');

			modalBody.append(iframe);
			modalBody.append(closeBtn);
			modalBody.append(modalLoader);

			var modalContent = document.createElement('div');

			modalContent.classList.add('vme-modal-content');

			modalContent.append(modalBody);

			var modalDialog = document.createElement('div');

			modalDialog.classList.add('vme-modal-dialog');

			modalDialog.append(modalContent);

			app.modal = document.createElement('div');

			app.modal.classList.add('vme-modal');
			app.modal.setAttribute('aria-modal', 'true');
			app.modal.setAttribute('role', 'dialog');
			app.modal.iframe = iframe;

			app.modal.append(modalDialog);

			document.body.append(app.modal);
		}

		if (!app.backdrop) {
			app.backdrop = document.createElement('div');

			app.backdrop.classList.add('vme-backdrop');

			document.body.append(app.backdrop);
		}

		app.startLoading();

		app.modal.iframe.src = link.href;
		app.modal.classList.add('vme-fade');
		app.modal.style.display = 'block';

		app.backdrop.classList.add('vme-fade');
		app.backdrop.style.display = 'block';

		if (app.timeout) {
			clearTimeout(app.timeout);
		}

		app.timeout = setTimeout(function() {
			app.modal.classList.add('vme-in');
			app.backdrop.classList.add('vme-in');
		}, 10);

		app.isOpen = true;
	}

	app.hide = function() {
		if (app.modal) {
			app.modal.classList.remove('vme-in');

			app.modal.addEventListener('transitionend', app._hide);
		}

		if (app.backdrop) {
			app.backdrop.classList.remove('vme-in');
		}
	}

	app._hide = function() {
		if (app.modal) {
			app.modal.style.display = 'none';
			app.modal.classList.remove('vme-fade');
		}

		if (app.backdrop) {
			app.backdrop.style.display = 'none';
			app.backdrop.classList.remove('vme-fade');
		}

		app.isOpen = false;

		app.modal.querySelector('iframe').src = 'about:blank';

		app.modal.removeEventListener('transitionend', app._hide);
	}

	app.startLoading = function() {
		if (!app.modal) {
			return false;
		}

		app.modal.querySelector('.vme-modal-body').classList.add('vme-modal-loading');
	}

	app.stopLoading = function() {
		if (!app.modal) {
			return false;
		}

		app.modal.querySelector('.vme-modal-body').classList.remove('vme-modal-loading');
	}

	app.setHeight = function(height) {
		if (!app.modal) {
			return false;
		}

		app.modal.querySelector('iframe').style.height = height ? height + 'px' : '';
	}

	document.addEventListener('vme.iframechanged', function(event) {
		app.startLoading();
	})

	document.addEventListener('DOMContentLoaded', function() {
		document.documentElement.addEventListener('click', function(e) {
			if (e.target && (e.target.matches(app.selector) || e.target.closest(app.selector))) {
				e.preventDefault();

				var link = e.target.matches(app.selector) ? e.target : e.target.closest(app.selector);

				app.show(link);

				return false;
			}

			if (e.target && app.isOpen && app.modal && !app.modal.querySelector('.vme-modal-content').contains(e.target)) {
				app.hide();
			}
		})
	})

	document.addEventListener('keydown', function(evt) {
		evt = evt || window.event;
		var isEscape = false;

		if ('key' in evt) {
			isEscape = (evt.key === 'Escape' || evt.key === 'Esc');
		} else {
			isEscape = (evt.keyCode === 27);
		}

		var isNotCombinedKey = !(evt.ctrlKey || evt.altKey || evt.shiftKey);

		if (app.isOpen && isEscape && isNotCombinedKey) {
			app.hide();
		}

		if (window.parent && window.location != window.parent.location && window.parent.vmeIframeModal && window.parent.vmeIframeModal.isOpen && isEscape && isNotCombinedKey) {
			window.parent.vmeIframeModal.hide();
		}
	})

	window.addEventListener('unload', function(event) {
		if (window.parent && window.location != window.parent.location) {
			var iframeSrcChangeEvent = new CustomEvent('vme.iframechanged', {
				details: event
			});

			window.parent.document.dispatchEvent(iframeSrcChangeEvent);
		}
	})

	window.addEventListener('resize', function() {
		if (app.isOpen && app.modal) {
			app.setHeight('');
			app.setHeight(app.modal.querySelector('iframe').contentWindow.document.body.offsetHeight);
		}
	})
})(vmeIframeModal, document, window);