<?php
/**
 * @package    Essentials for VirtueMart
 * @author     Abhishek Das <info@virtueplanet.com>
 * @copyright  Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://www.virtueplanet.com
 */

defined('_JEXEC') or die;

JLoader::registerPrefix('VMEssentials', JPATH_SITE . '/components/com_vmessentials/');
JLoader::register('VMEssentialsHelperUtility', JPATH_SITE . '/components/com_vmessentials/helpers/utility.php');

/**
 * Class VmessentialsRouter
 *
 * @since  3.3
 */
class VMEssentialsRouter extends JComponentRouterBase
{
	/**
	 * Build method for URLs
	 * This method is meant to transform the query parameters into a more human
	 * readable form. It is only executed when SEF mode is switched on.
	 *
	 * @param   array  &$query  An array of URL arguments
	 *
	 * @return  array  The URL arguments to use to assemble the subsequent URL.
	 *
	 * @since   3.3
	 */
	public function build(&$query)
	{
		$segments = array();
		$view     = null;

		// We need a menu item.  Either the one specified in the query, or the current active one if none specified
		if (empty($query['Itemid']))
		{
			$menuItem = $this->menu->getActive();
			$menuItemGiven = false;
		}
		else
		{
			$menuItem = $this->menu->getItem($query['Itemid']);
			$menuItemGiven = true;
		}
		
		// Check again
		if ($menuItemGiven && isset($menuItem) && $menuItem->component != 'com_vmessentials')
		{
			$menuItemGiven = false;
			unset($query['Itemid']);
		}

		if (isset($query['view']))
		{
			$view = $query['view'];
		}
		else
		{
			// We need to have a view in the query or it is an invalid URL
			return $segments;
		}

		if (($menuItem instanceof stdClass || $menuItem instanceof \Joomla\CMS\Menu\MenuItem) && $menuItem->query['view'] == $query['view'])
		{
			if (!empty($query['id']))
			{
				$segments[] = $query['id'];
			}
			
			unset($query['view']);
			unset($query['id']);

			return $segments;
		}

		if (isset($query['task']))
		{
			$taskParts  = explode('.', $query['task']);
			$segments[] = implode('/', $taskParts);
			$view       = $taskParts[0];
			unset($query['task']);
		}

		if (isset($query['view']))
		{
			$segments[] = $query['view'];
			$view = $query['view'];
			
			unset($query['view']);
		}

		if (isset($query['id']))
		{
			if ($view !== null)
			{
				$segments[] = $query['id'];
			}
			else
			{
				$segments[] = $query['id'];
			}

			unset($query['id']);
		}

		return $segments;
	}

	/**
	 * Parse method for URLs
	 * This method is meant to transform the human readable URL back into
	 * query parameters. It is only executed when SEF mode is switched on.
	 *
	 * @param   array  &$segments  The segments of the URL to parse.
	 *
	 * @return  array  The URL attributes to be used by the application.
	 *
	 * @since   3.3
	 */
	public function parse(&$segments)
	{
		$vars = array();

		// View is always the first element of the array
		$vars['view'] = array_shift($segments);
		$validViews   = array('compare', 'wishlist', 'quickview');
		
		if (!empty($segments))
		{
			while (!empty($segments))
			{
				$segment = array_pop($segments);

				// If it's the ID, let's put on the request
				if (is_numeric($segment) || VMEssentialsHelperUtility::isValidEncodedString($segment))
				{
					$vars['id'] = $segment;
				}
				else
				{
					$vars['task'] = $vars['view'] . '.' . $segment;
				}
			}
		}
		elseif (!in_array($vars['view'], $validViews) && VMEssentialsHelperUtility::isValidEncodedString($vars['view']))
		{
			$vars['id'] = $vars['view'];
			
			unset($vars['view']);
			
			$menus = JFactory::getApplication()->getMenu();
			$menu  = $menus->getActive();
			
			if ($menu)
			{
				if (isset($menu->query['option']) && $menu->query['option'] === 'com_vmessentials' && !empty($menu->query['view']))
				{
					$vars['view'] = $menu->query['view'];
				}
			}
		}

		return $vars;
	}
}
