<?php
/**
 * @package    Essentials for VirtueMart
 * @author     Abhishek Das <info@virtueplanet.com>
 * @copyright  Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://www.virtueplanet.com
 */

defined('_JEXEC') or die;

use \Joomla\CMS\Factory;

/**
 * Methods supporting a list of Wishlist records.
 *
 * @since  1.6
 */
class VMEssentialsModelCompare extends JModelLegacy
{
	protected $context = 'com_vmessentials.compare';
	
	/**
	 * Method to auto-populate the state.
	 *
	 * This method should only be called once per instantiation and is designed
	 * to be called on the first call to the getState() method unless the
	 * configuration flag to ignore the request is set.
	 *
	 * @return  void
	 *
	 * @note    Calling getState in this method will result in recursion.
	 */
	protected function populateState()
	{
		$input = Factory::getApplication()->input;
		
		$id = $input->get('id', '', VMEssentialsHelperUtility::getEncodeFilter());
		$this->setState('id', $id);
	}
	
	protected function getStoreId($id = '')
	{
		// Add the list state to the store id.
		$id .= ':' . Factory::getUser()->get('id');

		return md5($this->context . ':' . $id);
	}

	public function add($virtuemart_product_id)
	{
		$virtuemart_product_id = (int) $virtuemart_product_id;
		
		if (!is_numeric($virtuemart_product_id) || $virtuemart_product_id <= 0)
		{
			$this->setError(JText::_('COM_VMESSENTIALS_ERROR_INVALID_PRODUCT_ID'));

			return false;
		}
		
		$app    = Factory::getApplication();
		$user   = Factory::getUser();
		$params = $app->getParams('com_vmessentials');
		$items  = (array) $this->getAllItemIds();
		
		if (!empty($items))
		{
			if (in_array($virtuemart_product_id, $items))
			{
				// Item exists in compare
				return true;
			}
			
			$limit = $params->get('compare_max_items', 4);
			
			if (count($items) >= $limit)
			{
				$this->setError(JText::sprintf('COM_VMESSENTIALS_ERROR_COMPARE_MAX_LIMIT', $limit));

				return false;
			}
		}
		
		$session = Factory::getSession();
		$items[] = $virtuemart_product_id;
		
		$session->set('compare', $items, 'vmessentials');
		
		$this->cleanCache();
		
		return true;
	}
	
	public function remove($virtuemart_product_id)
	{
		$virtuemart_product_id = (int) $virtuemart_product_id;
		
		if (!is_numeric($virtuemart_product_id) || $virtuemart_product_id <= 0)
		{
			$this->setError(JText::_('COM_VMESSENTIALS_ERROR_INVALID_PRODUCT_ID'));

			return false;
		}
		
		$app    = Factory::getApplication();
		$user   = Factory::getUser();
		$params = $app->getParams('com_vmessentials');
		$items  = (array) $this->getAllItemIds();

		if (empty($items) || !in_array($virtuemart_product_id, $items))
		{
			// Item does not exists in the compare
			$this->setError(JText::_('COM_VMESSENTIALS_ERROR_INVALID_PRODUCT_ID'));

			return true;
		}

		$session = Factory::getSession();
		$key = array_search($virtuemart_product_id, $items);
		
		if ($key !== false)
		{
			unset($items[$key]);
			
			$session->set('compare', $items, 'vmessentials');
		}
		
		$this->cleanCache('com_vmessentials');
		$this->cleanCache('com_virtuemart');
		$this->cleanCache('mod_vp_product_carousel');
		$this->cleanCache('mod_virtuemart_product');
		
		return true;
	}
	
	public function getAllItemIds()
	{
		// Get a storage key.
		$store = $this->getStoreId('allItems');
		
		// Try to load the data from internal storage.
		if (isset($this->cache[$store]))
		{
			return $this->cache[$store];
		}
		
		$id    = $this->getState('id', '');
		$hasId = false;
		
		if (!empty($id))
		{
			$productIdsStr = VMEssentialsHelperUtility::decode($id);
			
			if ($productIdsStr)
			{
				$productIds          = strpos($productIdsStr, ',') === false ? array($productIdsStr) : explode(',', $productIdsStr);
				$this->cache[$store] = array_map('intval', $productIds);
				
				$hasId = true;
			}
		}
		
		if (!$hasId)
		{
			$session = Factory::getSession();
			
			$this->cache[$store] = (array) $session->get('compare', array(), 'vmessentials');
			$this->cache[$store] = !empty($this->cache[$store]) ? array_map('intval', $this->cache[$store]) : $this->cache[$store];
		}
		
		$this->setState('has_id', $hasId);
		
		return $this->cache[$store];
	}

	public function getItems()
	{
		// Get a storage key.
		$store      = $this->getStoreId();
		$fieldStore = $this->getStoreId('fields');
		
		// Try to load the data from internal storage.
		if (isset($this->cache[$store]))
		{
			return $this->cache[$store];
		}
		
		$virtuemart_product_ids = $this->getAllItemIds();
		$comparisonFields = array();
		
		if (is_array($virtuemart_product_ids) && !empty($virtuemart_product_ids))
		{
			$app    = Factory::getApplication();
			$params = $app->getParams('com_vmessentials');
			$compare_fields = (array) $params->get('compare_fields', array());
			$model = VmModel::getModel('Product');
			$ratingModel = VmModel::getModel('ratings');
			$model->withRating = $ratingModel->showRating();
			$products = $model->getProducts($virtuemart_product_ids);
			$model->addImages($products, 1);

			if (!empty($products))
			{
				JLoader::register('vmCustomPlugin', JPATH_VM_PLUGINS . '/vmcustomplugin.php');
				
				$customfieldsModel = VmModel::getModel('Customfields');
				
				foreach ($products as $key => $productItem)
				{
					$productItem->stock = $model->getStockIndicator($productItem);

					if (!empty($productItem->customfields))
					{
						$product      = clone($productItem);
						$customfields = array();
						
						foreach ($productItem->customfields as $cu)
						{
							$customfields[] = clone($cu);
						}
						
						$customfieldsGrouped = array();
						$customfieldsSorted = array();
						
						$customfieldsModel->displayProductCustomfieldFE($product, $customfields);
						//$product->stock = self::getStockText($productItem);

						foreach ($customfields as $k => $custom)
						{
							if (!empty($custom->virtuemart_custom_id) && !empty($custom->custom_title) && !in_array($custom->field_type, array('G', 'R', 'Z')) &&
							   !$custom->is_cart_attribute && !$custom->is_input && in_array($custom->virtuemart_custom_id, $compare_fields))
							{
								if (!isset($comparisonFields[$custom->virtuemart_custom_id]))
								{
									$comparisonFields[$custom->virtuemart_custom_id] = vmText::_($custom->custom_title);
								}
								
								if (!isset($customfieldsGrouped[$custom->virtuemart_custom_id]))
								{
									$customfieldsGrouped[$custom->virtuemart_custom_id] = array();
								}
								
								$customfieldsGrouped[$custom->virtuemart_custom_id][] = $custom;
							}
							
							if (!empty($custom->layout_pos))
							{
								if (!isset($customfieldsSorted[$custom->layout_pos]))
								{
									$customfieldsSorted[$custom->layout_pos] = array();
								}
								
								$customfieldsSorted[$custom->layout_pos][] = $custom;

								unset($customfields[$k]);
							}
						}
						
						$customfieldsSorted['normal'] = $customfields;
						$product->customfieldsSorted  = $customfieldsSorted;
						$product->customfieldsGrouped = $customfieldsGrouped;

						unset($product->customfields);

						$products[$key] = $product;
					} 
					else
					{
						$products[$key] = $productItem;
					}
				}
			}
			
			$this->cache[$store]      = $products;
			$this->cache[$fieldStore] = $comparisonFields;
		}
		else
		{
			$this->cache[$store]      = array();
			$this->cache[$fieldStore] = array();
		}
		
		return $this->cache[$store];
	}
	
	public function getCompareFields()
	{
		$fieldStore = $this->getStoreId('fields');
		
		return isset($this->cache[$fieldStore]) ? $this->cache[$fieldStore] : array();
	}
	
	public function getPermalink()
	{
		$productIds = $this->getAllItemIds();
		
		if (empty($productIds))
		{
			return false;
		}
		
		$productIds     = array_map('intval', $productIds);
		$productIdsStr  = VMEssentialsHelperUtility::encode(implode(',', $productIds));
		$camparePageUrl = VMEssentialsHelperRoute::getRoute('compare') . '&id=' . $productIdsStr;
		
		return $camparePageUrl;
	}
	
	protected function cleanCache($group = null, $client_id = 0)
	{
		$conf = Factory::getConfig();

		$options = array(
			'defaultgroup' => $group,
			'cachebase'    => ($client_id) ? JPATH_ADMINISTRATOR . '/cache' : $conf->get('cache_path', JPATH_SITE . '/cache')
		);

		$cache = JCache::getInstance('callback', $options);

		$cache->clean();

		// Trigger the onContentCleanCache event.
		if (version_compare(JVERSION, '4.0.0', 'ge'))
		{
			Factory::getApplication()->triggerEvent('onExtensionCleanCache', $options);
		}
		else
		{
			JEventDispatcher::getInstance()->trigger('onExtensionCleanCache', $options);
		}
	}
}
