<?php
/**
 * @package    Essentials for VirtueMart
 * @author     Abhishek Das <info@virtueplanet.com>
 * @copyright  Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://www.virtueplanet.com
 */

defined('_JEXEC') or die;

/**
 * Class VMEssentials Frontend Helper
 *
 * @since  1.6
 */
class VMEssentialsHelpersVMEssentials
{
	/**
	 * Get an instance of the named model
	 *
	 * @param   string  $name  Model name
	 *
	 * @return null|object
	 */
	public static function getModel($name, $config = array('ignore_request' => true))
	{
		jimport('legacy.model.legacy');

		JModelLegacy::addIncludePath(VMESSENTIALS_SITE . '/models');
		
		return JModelLegacy::getInstance($name, 'VMEssentialsModel', $config);
	}

	/**
	* Gets the edit permission for an user
	*
	* @param   mixed  $item  The item
	*
	* @return  bool
	*/
	public static function canUserEdit($item)
	{
		$permission = false;
		$user       = JFactory::getUser();

		if ($user->authorise('core.edit', 'com_vmessentials'))
		{
			$permission = true;
		}
		else
		{
			if (isset($item->created_by))
			{
				if ($user->authorise('core.edit.own', 'com_vmessentials') && $item->created_by == $user->id)
				{
					$permission = true;
				}
			}
			else
			{
				$permission = true;
			}
		}

		return $permission;
	}
	
	public static function validateProductIds($virtuemart_product_ids)
	{
		if (empty($virtuemart_product_ids) || !is_array($virtuemart_product_ids))
		{
			return array();
		}
		
		$app   = JFactory::getApplication();
		$db    = JFactory::getDbo();
		$query = $db->getQuery(true);
		
		$query->select($db->quoteName('virtuemart_product_id'))
			->from($db->quoteName('#__virtuemart_products'))
			->where($db->quoteName('virtuemart_product_id') . ' IN (' . implode(',', $virtuemart_product_ids) . ')');
		
		$db->setQuery($query);
		$ids = $db->loadColumn();
		
		return $ids;
	}
	
	public static function moveWishlistToUser()
	{
		$user    = JFactory::getUser();
		$user_id = (int) $user->get('id');
		
		if ($user->get('guest') || empty($user_id))
		{
			return true;
		}
		
		jimport('legacy.model.legacy');
		
		JModelLegacy::addIncludePath(VMESSENTIALS_SITE . '/models');
		
		$model      = JModelLegacy::getInstance('Wishlist', 'VMEssentialsModel', array('ignore_request' => true));
		$guestItems = $model->getGuestItems();
		
		if (empty($guestItems))
		{
			return true;
		}
		
		$app     = JFactory::getApplication();
		$db      = JFactory::getDbo();
		$query   = $db->getQuery(true);

		// Select the required fields from the table.
		$query->select($db->quoteName('virtuemart_product_id'))
			->from($db->quoteName('#__vmessentials_wishlist'))
			->where($db->quoteName('user_id') . ' = ' . $user_id);
		
		$db->setQuery($query);
		
		try 
		{
			$items = $db->loadColumn();
		}
		catch (Exception $e) 
		{
			$app->enqueueMessage($e->getMessage(), 'error');
			return false;
		}
		
		if (!empty($items))
		{
			$guestItems = array_diff(array_map('intval', $guestItems), array_map('intval', $items));
		}
		
		$guestItems = array_filter($guestItems);
		
		if (empty($guestItems))
		{
			return true;
		}
		
		$date = JFactory::getDate()->toSql();
		
		$query->clear()
			->insert($db->quoteName('#__vmessentials_wishlist'))
			->columns(array($db->quoteName('user_id'), $db->quoteName('virtuemart_product_id'), $db->quoteName('state'), $db->quoteName('created_by'), $db->quoteName('datetime')));
		
		foreach ($guestItems as $virtuemart_product_id)
		{
			$query->values($user_id . ',' . $virtuemart_product_id . ',1,' . $user_id . ',' . $db->quote($date));
		}
		
		$db->setQuery($query);
		
		try 
		{
			$db->execute();
		}
		catch (Exception $e) 
		{
			$app->enqueueMessage($e->getMessage(), 'error');
			return false;
		}
		
		$model->clearGuestList();
		$model->resetCache();
		
		return true;
	}
}
