<?php
/**
 * @package    Essentials for VirtueMart
 * @author     Abhishek Das <info@virtueplanet.com>
 * @copyright  Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://www.virtueplanet.com
 */

defined('_JEXEC') or die;

/**
 * VMEssentials helper.
 *
 * @since  1.6
 */
class VMEssentialsHelper
{
	protected static $extensions;
	
	
	public static function checkSystem()
	{
		$app = JFactory::getApplication();
		
		if (!self::hasExtension('vmessentials', 'plugin', 'system'))
		{
			$url = JRoute::_('index.php?option=com_plugins&view=plugins&filter_search=essentials+for+virtuemart');
			$link = ' <a href="' . $url . '" target="_blank">System - Essentials for VirtueMart</a>';
			
			$app->enqueueMessage(JText::sprintf('COM_VMESSENTIALS_WARNING_REQUIRED_PLUGINS_MISSING', $link), 'warning');
		}
	}
	
	/**
	* Method to check if an extension installed and enabled
	* 
	* @param  string  $element  Name of the extension
	* @param  string  $type     Type of the extension like component, module, plugins etc
	* 
	* @return boolean
	*/
	public static function hasExtension($element, $type = 'component', $folder = null)
	{
		if (!is_array(self::$extensions))
		{
			self::$extensions = array();
		}
		
		$hash = 'element:' . $element . '.type:' . $type;
		
		if (!empty($folder))
		{
			$hash .= '.folder:' . $folder;
		}
		
		if (!array_key_exists($hash, self::$extensions))
		{
			$type = strtolower($type);
			self::$extensions[$hash] = false;
			
			$db = JFactory::getDbo();
			$query = $db->getQuery(true)
			            ->select('extension_id, element, folder')
			            ->from('#__extensions')
			            ->where('type = ' . $db->quote($type))
			            ->where('element = ' . $db->quote($element))
			            ->where('enabled = 1');
			
			try
			{
				$db->setQuery($query);
				$results = $db->loadObjectList();
			}
			catch(Exception $e)
			{
				JFactory::getApplication()->enqueueMessage($e->getMessage(), 'error');
				return false;
			}
			
			if (!empty($results))
			{
				foreach ($results as $result)
				{
					if ($type != 'package')
					{
						switch($type)
						{
							case 'module':
								$path = JPATH_SITE . '/modules/' . $result->element;
								break;
							case 'plugin':
								if (!empty($folder) && strtolower($folder) != $result->folder)
								{
									$path = null;
								}
								else
								{
									$path = JPATH_SITE . '/plugins/' . $result->folder . '/' . $result->element;
								}
								break;
							default:
							case 'component':
								$path = JPATH_ADMINISTRATOR . '/components/' . $result->element;
						}
						
						if ($path && file_exists($path) && is_dir($path))
						{
							self::$extensions[$hash] = true;
							break;
						}
					}
					else
					{
						self::$extensions[$hash] = true;
						break;
					}
				}
			}
		}

		return self::$extensions[$hash];
	}
	
	/**
	 * Configure the Linkbar.
	 *
	 * @param   string  $vName  string
	 *
	 * @return void
	 */
	public static function addSubmenu($vName = '')
	{
		JHtmlSidebar::addEntry(
			JText::_('COM_VMESSENTIALS_TITLE_ITEMS'),
			'index.php?option=com_vmessentials&view=items',
			$vName == 'items'
		);

		JHtmlSidebar::addEntry(
			JText::_('COM_VMESSENTIALS_CONFIGURATION'),
			'index.php?option=com_config&view=component&component=com_vmessentials',
			$vName == 'component'
		);
	}

	/**
	 * Gets the files attached to an item
	 *
	 * @param   int     $pk     The item's id
	 *
	 * @param   string  $table  The table's name
	 *
	 * @param   string  $field  The field's name
	 *
	 * @return  array  The files
	 */
	public static function getFiles($pk, $table, $field)
	{
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);

		$query
			->select($field)
			->from($table)
			->where('id = ' . (int) $pk);

		$db->setQuery($query);

		return explode(',', $db->loadResult());
	}

	/**
	 * Gets a list of the actions that can be performed.
	 *
	 * @return    JObject
	 *
	 * @since    1.6
	 */
	public static function getActions()
	{
		$user   = JFactory::getUser();
		$result = new JObject;

		$assetName = 'com_vmessentials';

		$actions = array(
			'core.admin', 'core.manage', 'core.create', 'core.edit', 'core.edit.own', 'core.edit.state', 'core.delete'
		);

		foreach ($actions as $action)
		{
			$result->set($action, $user->authorise($action, $assetName));
		}

		return $result;
	}
}
