<?php
/**
 * @package    Essentials for VirtueMart
 * @author     Abhishek Das <info@virtueplanet.com>
 * @copyright  Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://www.virtueplanet.com
 */

defined('_JEXEC') or die;

use Joomla\Utilities\ArrayHelper;

/**
 * Items list controller class.
 *
 * @since  1.6
 */
class VMEssentialsControllerItems extends JControllerAdmin
{
	/**
	 * Removes an item.
	 *
	 * @return  void
	 *
	 * @since   12.2
	 */
	public function delete()
	{
		// Check for request forgeries
		JSession::checkToken() or die(JText::_('JINVALID_TOKEN'));
		
		$app = JFactory::getApplication();
		$input = $app->input;
		
		// Get items to remove from the request.
		$cid = $input->get('cid', array(), 'array');

		if (!is_array($cid) || count($cid) < 1)
		{
			JLog::add(JText::_($this->text_prefix . '_NO_ITEM_SELECTED'), JLog::WARNING, 'jerror');
		}
		else
		{
			$by_product = $input->get('byproduct', 0, 'INT');
			
			// Get the model.
			$model = $this->getModel();

			// Make sure the item ids are integers
			jimport('joomla.utilities.arrayhelper');
			ArrayHelper::toInteger($cid);

			// Remove the items.
			if ($model->delete($cid, $by_product))
			{
				$this->setMessage(JText::plural($this->text_prefix . '_N_ITEMS_DELETED', count($cid)));
			}
			else
			{
				$this->setMessage($model->getError(), 'error');
			}

			// Invoke the postDelete method to allow for the child class to access the model.
			$this->postDeleteHook($model, $cid);
		}

		$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
	}
	
	public function download()
	{
		// Check for request forgeries
		JSession::checkToken('GET') or die(JText::_('JINVALID_TOKEN'));
		
		$model = $this->getModel();
		
		if (!$data = $model->getAllItems())
		{
			$this->setMessage($model->getError(), 'error');
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
			
			return false;
		}
		
		$output = array();
		
		if (!empty($data))
		{
			$keys = array_keys($data[0]);
			
			$separator = ',';
			$data = array_merge(array($keys), $data);

			foreach ($data as $item)
			{
				foreach ($item as $value)
				{
					$value = str_replace(array($separator, '"'), array(' ', ''), $value);
					$output[] = $value . $separator;
				}
				
				$output[] = PHP_EOL;
			}
		}
		
		$output = !empty($output) ? implode($output) : '';
		
		$this->returnCSV($output);
	}
	
	protected function returnCSV($output)
	{
		$app     = JFactory::getApplication();
		$obLevel = ob_get_level();
		
		if ($obLevel)
		{
			while ($obLevel > 0)
			{
				ob_end_clean();
				$obLevel --;
			}
		}
		elseif (ob_get_contents())
		{
			ob_clean();
		}
		
		header('Content-type: application/octet-stream');
		header('Content-Disposition: attachment; filename=wishlist.csv');
		header('Cache-Control: public,max-age=1,must-revalidate');
		header('Expires: ' . gmdate('D, d M Y H:i:s', ($_SERVER['REQUEST_TIME'] + 1)) . ' GMT');
		header('Last-modified: ' . gmdate('D, d M Y H:i:s', $_SERVER['REQUEST_TIME']) . ' GMT');

		if (function_exists('header_remove')) 
		{
			header_remove('Pragma');
		}
		
		echo $output;
		
		flush();
		$app->close();
	}
	
	/**
	 * Proxy for getModel.
	 *
	 * @param   string  $name    Optional. Model name
	 * @param   string  $prefix  Optional. Class prefix
	 * @param   array   $config  Optional. Configuration array for model
	 *
	 * @return  object	The Model
	 *
	 * @since    1.6
	 */
	public function getModel($name = 'Items', $prefix = 'VMEssentialsModel', $config = array('ignore_request' => true))
	{
		return parent::getModel($name, $prefix, $config);
	}
}
